from http import HTTPStatus
from typing import Any, Optional, Union

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.get_canvas_response import GetCanvasResponse
from ...models.http_validation_error import HTTPValidationError
from ...types import UNSET, Response


def _get_kwargs(
    *,
    canvas_id: str,
) -> dict[str, Any]:
    params: dict[str, Any] = {}

    params["canvas_id"] = canvas_id

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": "/api/v1/canvas",
        "params": params,
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[GetCanvasResponse, HTTPValidationError]]:
    if response.status_code == 200:
        response_200 = GetCanvasResponse.from_dict(response.json())

        return response_200
    if response.status_code == 422:
        response_422 = HTTPValidationError.from_dict(response.json())

        return response_422
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[GetCanvasResponse, HTTPValidationError]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    canvas_id: str,
) -> Response[Union[GetCanvasResponse, HTTPValidationError]]:
    """Get Canvas

     Get a full canvas by ID.
    Args:
        canvas_id: Canvas UUID to retrieve
    Returns:
        CanvasData on success
    Raises:
        HTTPException: 404 if canvas not found

    Args:
        canvas_id (str): Canvas UUID

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[GetCanvasResponse, HTTPValidationError]]
    """

    kwargs = _get_kwargs(
        canvas_id=canvas_id,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    *,
    client: Union[AuthenticatedClient, Client],
    canvas_id: str,
) -> Optional[Union[GetCanvasResponse, HTTPValidationError]]:
    """Get Canvas

     Get a full canvas by ID.
    Args:
        canvas_id: Canvas UUID to retrieve
    Returns:
        CanvasData on success
    Raises:
        HTTPException: 404 if canvas not found

    Args:
        canvas_id (str): Canvas UUID

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[GetCanvasResponse, HTTPValidationError]
    """

    return sync_detailed(
        client=client,
        canvas_id=canvas_id,
    ).parsed


async def asyncio_detailed(
    *,
    client: Union[AuthenticatedClient, Client],
    canvas_id: str,
) -> Response[Union[GetCanvasResponse, HTTPValidationError]]:
    """Get Canvas

     Get a full canvas by ID.
    Args:
        canvas_id: Canvas UUID to retrieve
    Returns:
        CanvasData on success
    Raises:
        HTTPException: 404 if canvas not found

    Args:
        canvas_id (str): Canvas UUID

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[GetCanvasResponse, HTTPValidationError]]
    """

    kwargs = _get_kwargs(
        canvas_id=canvas_id,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    *,
    client: Union[AuthenticatedClient, Client],
    canvas_id: str,
) -> Optional[Union[GetCanvasResponse, HTTPValidationError]]:
    """Get Canvas

     Get a full canvas by ID.
    Args:
        canvas_id: Canvas UUID to retrieve
    Returns:
        CanvasData on success
    Raises:
        HTTPException: 404 if canvas not found

    Args:
        canvas_id (str): Canvas UUID

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[GetCanvasResponse, HTTPValidationError]
    """

    return (
        await asyncio_detailed(
            client=client,
            canvas_id=canvas_id,
        )
    ).parsed
