"""
Coercion functions for pre-validation data transformation.
These functions are applied before Pydantic validation (mode='before').
"""

from typing import Any, Callable, Dict

from pycharter.coercions.builtin import (
    coerce_to_boolean,
    coerce_to_datetime,
    coerce_to_float,
    coerce_to_integer,
    coerce_to_lowercase,
    coerce_to_string,
    coerce_to_uuid,
)

# Registry of available coercion functions
COERCION_REGISTRY: Dict[str, Callable[[Any], Any]] = {
    "coerce_to_string": coerce_to_string,
    "coerce_to_integer": coerce_to_integer,
    "coerce_to_float": coerce_to_float,
    "coerce_to_boolean": coerce_to_boolean,
    "coerce_to_datetime": coerce_to_datetime,
    "coerce_to_uuid": coerce_to_uuid,
    "coerce_to_lowercase": coerce_to_lowercase,
}


def get_coercion(name: str) -> Callable[[Any], Any]:
    """
    Get a coercion function by name.
    
    Args:
        name: Name of the coercion function
        
    Returns:
        The coercion function
        
    Raises:
        ValueError: If coercion function not found
    """
    if name not in COERCION_REGISTRY:
        raise ValueError(
            f"Coercion function '{name}' not found. "
            f"Available: {list(COERCION_REGISTRY.keys())}"
        )
    return COERCION_REGISTRY[name]


def register_coercion(name: str, func: Callable[[Any], Any]) -> None:
    """
    Register a custom coercion function.
    
    Args:
        name: Name to register the function under
        func: The coercion function
    """
    COERCION_REGISTRY[name] = func

