"""
Utility functions for PAB
"""
from contextlib import contextmanager
import os
import sys

import click
from colorama import Fore, Style, init

# Initialize colorama for Windows support
init()


def print_success(message):
    """Print success message in green"""
    print(f"{Fore.GREEN} {message}{Style.RESET_ALL}")


def print_error(message):
    """Print error message in red"""
    print(f"{Fore.RED} {message}{Style.RESET_ALL}", file=sys.stderr)


def print_info(message, end='\n'):
    """Print info message in blue"""
    print(f"{Fore.BLUE} {message}{Style.RESET_ALL}", end=end)


def print_warning(message):
    """Print warning message in yellow"""
    print(f"{Fore.YELLOW} {message}{Style.RESET_ALL}")


def print_cyan(message):
    """Print normal message in cyan"""
    print(f"{Fore.CYAN} {message}{Style.RESET_ALL}")


TEMPLATE = """\
# Automatically generated by: pab deploy

from setuptools import setup, find_packages

setup(
    name='project',
    version='0.1.0',
    description='APCloudy Deployment Tool for Scrapy Spiders and Scripts',
    author='Fawad',
    author_email='fawadstar6@gmail.com',
    packages=find_packages(),
    scripts=[],
    entry_points={'scrapy': ['settings = %(settings)s']},
)
"""


@contextmanager
def remember_cwd():
    current_dir = os.getcwd()
    try:
        yield
    finally:
        os.chdir(current_dir)


def find_file(filename, start_path='.'):
    """Find the file by traversing up the directory tree"""
    path = os.path.abspath(start_path)
    while True:
        file_path = os.path.join(path, filename)
        if os.path.exists(file_path):
            return file_path
        parent = os.path.dirname(path)
        if parent == path:
            return None
        path = parent


def parse_cfg_file(cfg_file):
    """Extract settings value from scrapy.cfg file"""
    try:
        with open(cfg_file) as f:
            for line in f:
                line = line.strip()
                if line.startswith('default ='):
                    return line.split('=', 1)[1].strip()
    except (IOError, OSError):
        pass
    return 'default'


def create_setup(**kwargs):
    scrapy_cfg = find_file('scrapy.cfg')
    if not scrapy_cfg:
        click.echo("No scrapy.cfg found")
        return

    with remember_cwd():
        os.chdir(os.path.dirname(scrapy_cfg))

        if os.path.exists('setup.py'):
            return

        if 'settings' not in kwargs:
            kwargs['settings'] = parse_cfg_file(scrapy_cfg)

        with open('setup.py', 'w') as f:
            f.write(TEMPLATE % kwargs)
