# Руководство по функции свечения линии (glow_line)

![Пример свечения линии](images/Glow.png)

## Описание

Функция `glow_line` создаёт эффект свечения для линий на графике, используя несколько полупрозрачных слоёв с увеличивающейся толщиной и уменьшающейся прозрачностью.

## Основные параметры

### Базовые параметры
- **`glow_color`** - Цвет свечения (по умолчанию цвет основной линии)
- **`glow_width`** - Ширина свечения как множитель от толщины основной линии (по умолчанию 3.0)
- **`glow_alpha`** - Базовая прозрачность свечения (0.0-1.0, по умолчанию 0.5)
- **`glow_layers`** - Количество слоёв свечения (по умолчанию 10)

### Функции затухания
- **`decay_function`** - Функция затухания f(distance_ratio) -> alpha_multiplier

Доступные встроенные функции затухания:
- `'linear'` - Линейное затухание
- `'exponential'` - Экспоненциальное затухание
- `'gaussian'` - Гауссово затухание (по умолчанию)
- `'quadratic'` - Квадратичное затухание
- `'cubic'` - Кубическое затухание
- `'sine'` - Синусоидальное затухание

### Режимы прозрачности
- **`alpha_mode`** - Режим изменения прозрачности:
  - `'uniform'` - Равномерная прозрачность (по умолчанию)
  - `'gradient'` - Градиентное уменьшение прозрачности
  - `'pulse'` - Пульсирующая прозрачность

### Цветовые схемы
- **`colormap`** - Название цветовой схемы matplotlib (например, 'plasma', 'viridis', 'inferno')

## Примеры использования

### Базовое использование
```python
import eleganplot
import numpy as np

fig, ax = eleganplot.subplots()
x = np.linspace(0, 10, 100)
y = np.sin(x)

# Простое свечение
main_line, glow_lines = ax.glow_line(x, y, glow_color='cyan', glow_width=5.0)
```

Результат выглядит так:

![Базовое свечение](images/Simple.png)

### Пользовательская функция затухания
```python
# Создаём кастомную функцию затухания
custom_decay = lambda r: np.exp(-5 * r**3)

main_line, glow_lines = ax.glow_line(
    x, y, 
    decay_function=custom_decay,
    glow_alpha=0.8
)
```

### Использование встроенных функций затухания
```python
from eleganplot.utils import get_decay_functions

decay_funcs = get_decay_functions()
exponential_decay = decay_funcs['exponential']

main_line, glow_lines = ax.glow_line(
    x, y,
    decay_function=exponential_decay,
    glow_width=4.0
)
```

### Цветовые схемы
```python
# Использование цветовой схемы plasma
main_line, glow_lines = ax.glow_line(
    x, y,
    colormap='plasma',
    glow_width=6.0,
    alpha_mode='gradient'
)
```

### Комплексный пример
```python
# Комбинирование всех возможностей
main_line, glow_lines = ax.glow_line(
    x, y,
    glow_color='orange',
    colormap='viridis',
    glow_width=8.0,
    glow_alpha=0.7,
    glow_layers=20,
    decay_function=lambda r: 1 / (1 + 3*r**2),
    alpha_mode='pulse'
)
```

## Советы по использованию

1. **Для тёмных фонов** используйте яркие цвета свечения (cyan, lime, orange)
2. **Для светлых фонов** используйте более тёмные цвета или уменьшите `glow_alpha`
3. **Для тонких эффектов** используйте меньше слоёв (5-8) и меньшую ширину
4. **Для драматических эффектов** увеличьте количество слоёв (15-25) и ширину
5. **Экспериментируйте** с различными функциями затухания для разных эффектов

## Производительность

- Большое количество слоёв может замедлить отрисовку
- Для интерактивных графиков рекомендуется 5-10 слоёв
- Для статических изображений можно использовать 15-25 слоёв

## Возвращаемые значения

Функция возвращает кортеж `(main_line, glow_lines)`:
- `main_line` - объект основной линии (Line2D)
- `glow_lines` - список объектов линий свечения (list[Line2D])

Это позволяет дополнительно настраивать свойства линий после создания.
