import json
import subprocess
import zipfile
from pathlib import Path
from tempfile import TemporaryDirectory
from typing import Optional

import click

from hafnia.log import sys_logger, user_logger
from hafnia_cli.config import Config


@click.group(name="runc")
def runc():
    """Creating and running trainer packages locally"""
    pass


@runc.command(name="launch-local")
@click.argument("exec_cmd", type=str)
@click.option(
    "--dataset",
    type=str,
    help="Hafnia dataset name e.g. mnist, midwest-vehicle-detection or a path to a local dataset",
    required=True,
)
@click.option(
    "--image_name",
    type=Optional[str],
    default=None,
    help=(
        "Docker image name to use for the launch. "
        "By default, it will use image name from '.state.json' "
        "file generated by the 'hafnia runc build-local' command"
    ),
)
@click.pass_obj
def launch_local(cfg: Config, exec_cmd: str, dataset: str, image_name: str) -> None:
    """Launch a job within the image."""
    from hafnia.platform.datasets import download_or_get_dataset_path

    is_local_dataset = "/" in dataset
    if is_local_dataset:
        click.echo(f"Using local dataset: {dataset}")
        path_dataset = Path(dataset)
        if not path_dataset.exists():
            raise click.ClickException(f"Dataset path does not exist: {path_dataset}")
    else:
        click.echo(f"Using Hafnia dataset: {dataset}")
        path_dataset = download_or_get_dataset_path(dataset_name=dataset, cfg=cfg, force_redownload=False)

    if image_name is None:
        # Load image name from state.json
        path_state_file = Path("state.json")
        if not path_state_file.exists():
            raise click.ClickException("State file does not exist. Please build the image first.")
        state_dict = json.loads(path_state_file.read_text())
        if "image_tag" not in state_dict:
            raise click.ClickException("'image_tag' not found in state file. Please build the image first.")
        image_name = state_dict["image_tag"]

    docker_cmds = [
        "docker",
        "run",
        "--rm",
        "-v",
        f"{path_dataset.absolute()}:/opt/ml/input/data/training",
        "-e",
        "HAFNIA_CLOUD=true",
        "-e",
        "PYTHONPATH=src",
        "--runtime",
        "nvidia",
        image_name,
    ] + exec_cmd.split(" ")

    # Use the "hafnia runc launch" cmd when we have moved to the new folder structure and
    # direct commands.
    # Replace '+ exec_cmd.split(" ")' with '["hafnia", "runc", "launch"] + exec_cmd.split(" ")'

    click.echo(f"Running command: \n\t{' '.join(docker_cmds)}")
    subprocess.run(docker_cmds, check=True)


@runc.command(name="build")
@click.argument("recipe_url")
@click.option("--state_file", "--st", type=str, default="state.json")
@click.option("--repo", type=str, default="localhost", help="Docker repository")
@click.pass_obj
def build(cfg: Config, recipe_url: str, state_file: str, repo: str) -> None:
    """Build docker image with a given recipe."""
    from hafnia.platform.builder import build_image, prepare_trainer_package

    with TemporaryDirectory() as temp_dir:
        metadata = prepare_trainer_package(recipe_url, Path(temp_dir), cfg.api_key)
        build_image(metadata, repo, state_file=state_file)


@runc.command(name="build-local")
@click.argument("recipe")
@click.option("--state_file", "--st", type=str, default="state.json")
@click.option("--repo", type=str, default="localhost", help="Docker repository")
def build_local(recipe: Path, state_file: str, repo: str) -> None:
    """Build recipe from local path as image with prefix - localhost"""
    import shutil
    import uuid

    import seedir

    from hafnia.platform.builder import build_image
    from hafnia.utils import filter_trainer_package_files

    recipe = Path(recipe)

    with TemporaryDirectory() as d:
        tmp_dir = Path(d)
        recipe_dir = tmp_dir / "recipe"
        recipe_dir.mkdir(parents=True, exist_ok=True)

        if recipe.suffix == ".zip":
            user_logger.info("Extracting recipe for processing.")
            with zipfile.ZipFile(recipe.as_posix(), "r") as zip_ref:
                zip_ref.extractall(recipe_dir)
        elif recipe.is_dir():
            for rf in filter_trainer_package_files(recipe):
                src_path = (recipe / rf).absolute()
                target_path = recipe_dir / rf
                target_path.parent.mkdir(parents=True, exist_ok=True)
                shutil.copyfile(src_path, target_path)

        user_logger.info(
            seedir.seedir(recipe_dir, sort=True, first="folders", style="emoji", printout=False, depthlimit=2)
        )

        metadata = {
            "dockerfile": (recipe_dir / "Dockerfile").as_posix(),
            "docker_context": recipe_dir.as_posix(),
            "digest": uuid.uuid4().hex[:8],
        }

        user_logger.info("Start building image.")
        sys_logger.debug(metadata)
        build_image(metadata, repo, state_file=state_file)
