from jitx.circuit import Circuit
from jitx.component import Component
from jitx.container import inline
from jitx.feature import Courtyard, Custom, Paste, Silkscreen, Soldermask
from jitx.interval import Interval
from jitx.landpattern import Landpattern, Pad, PadMapping
from jitx.net import Port
from jitx.shapes.composites import rectangle
from jitx.shapes.primitive import Anchor, Arc, ArcPolyline, Polygon, Polyline, Text
from jitx.si import BridgingPinModel
from jitx.symbol import Pin, Symbol, Direction, SymbolMapping
from jitxlib.parts import Capacitor
from jitxlib.parts.query_api import Inductor


class Pad1(Pad):
    shape = rectangle(2.2, 1.1)
    soldermask = Soldermask(rectangle(2.302, 1.202))
    paste = Paste(rectangle(2.302, 1.202))


class Pad2(Pad):
    shape = rectangle(1.05, 1)
    soldermask = Soldermask(rectangle(1.152, 1.102))
    paste = Paste(rectangle(1.152, 1.102))


class U_FL_R_SMT_10(Circuit):
    @inline
    class component(Component):
        """1 Inner needle IPEX Board Edge 1.25mm -40℃~+90℃ 6GHz 50Ω 2mm SMD  RF Connectors / Coaxial Connectors ROHS"""

        name = "u.FL Connector"
        manufacturer = "HIROSE ELECTRIC CO LTD"
        mpn = "U.FL-R-SMT(10)"
        datasheet = "https://datasheet.lcsc.com/lcsc/2304140030_HRS-Hirose-U-FL-R-SMT-10_C434808.pdf"
        reference_designator_prefix = "J"

        GND = [Port() for _ in range(2)]
        SIG = Port()

        @inline
        class symbol(Symbol):
            pin_name_size = 0.6
            pad_name_size = 0.6
            GND = [
                Pin(at=(-6, 0), length=4, direction=Direction.Left),
                Pin(at=(6, 0), length=4, direction=Direction.Right),
            ]
            SIG = Pin(at=(0, 6), length=4, direction=Direction.Up)

            value = Text(">VALUE", 0.5, Anchor.C).at(0, 6.5)
            reference = Text(">REF", 0.5, Anchor.C).at(0, 7.5)
            art = [
                rectangle(12, 8).at(0, 2),
            ]

        @inline
        class landpattern(Landpattern):
            p = {
                1: Pad1().at(-1.49, -0.312, rotate=90),
                2: Pad2().at(0.01, 1.188, rotate=90),
                3: Pad1().at(1.51, -0.312, rotate=90),
            }
            courtyard = Courtyard(rectangle(5.323, 3.478))
            labels = [
                Silkscreen(Text(">REF", 0.5, Anchor.W).at(-0.75, 3.445)),
                Custom(Text(">VALUE", 0.5, Anchor.W).at(-0.75, 2.445), name="Fab"),
            ]
            silkscreen = [
                Silkscreen(Polyline(0.254, [(0.847, -1.612), (-0.827, -1.612)])),
                Silkscreen(Polyline(0.254, [(0.741, 0.989), (0.847, 0.989)])),
                Silkscreen(Polyline(0.254, [(-0.827, 0.989), (-0.721, 0.989)])),
                Silkscreen(ArcPolyline(0.254, [Arc((0.01, -0.312), 0.224, 0, -360)])),
                Silkscreen(ArcPolyline(0.254, [Arc((0.01, -0.312), 0.781, 0, -360)])),
            ]
            fab_drawings = [
                Custom(shape, name="Fab")
                for shape in [
                    ArcPolyline(0.06, [Arc((-1.5, -1.742), 0.03, 0, -360)]),
                    Polygon(
                        [
                            (-2.555, -0.152),
                            (-2.555, -0.131),
                            (-2.384, -0.131),
                            (-2.384, -0.152),
                            (-2.555, -0.152),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.152),
                            (-2.299, -0.131),
                            (-2.192, -0.131),
                            (-2.192, -0.152),
                            (-2.299, -0.152),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.152),
                            (-2.064, -0.131),
                            (-1.978, -0.131),
                            (-1.978, -0.152),
                            (-2.064, -0.152),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.152),
                            (-1.936, -0.131),
                            (-1.722, -0.131),
                            (-1.722, -0.152),
                            (-1.936, -0.152),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.576, -0.173),
                            (-2.576, -0.152),
                            (-2.363, -0.152),
                            (-2.363, -0.173),
                            (-2.576, -0.173),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.173),
                            (-2.299, -0.152),
                            (-2.192, -0.152),
                            (-2.192, -0.173),
                            (-2.299, -0.173),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.173),
                            (-2.064, -0.152),
                            (-1.978, -0.152),
                            (-1.978, -0.173),
                            (-2.064, -0.173),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.173),
                            (-1.936, -0.152),
                            (-1.679, -0.152),
                            (-1.679, -0.173),
                            (-1.936, -0.173),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.619, -0.195),
                            (-2.619, -0.173),
                            (-2.512, -0.173),
                            (-2.512, -0.195),
                            (-2.619, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.427, -0.195),
                            (-2.427, -0.173),
                            (-2.363, -0.173),
                            (-2.363, -0.195),
                            (-2.427, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.195),
                            (-2.299, -0.173),
                            (-2.171, -0.173),
                            (-2.171, -0.195),
                            (-2.299, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.195),
                            (-2.064, -0.173),
                            (-1.978, -0.173),
                            (-1.978, -0.195),
                            (-2.064, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.195),
                            (-1.936, -0.173),
                            (-1.85, -0.173),
                            (-1.85, -0.195),
                            (-1.936, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.765, -0.195),
                            (-1.765, -0.173),
                            (-1.658, -0.173),
                            (-1.658, -0.195),
                            (-1.765, -0.195),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.619, -0.216),
                            (-2.619, -0.195),
                            (-2.534, -0.195),
                            (-2.534, -0.216),
                            (-2.619, -0.216),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.216),
                            (-2.299, -0.195),
                            (-2.149, -0.195),
                            (-2.149, -0.216),
                            (-2.299, -0.216),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.216),
                            (-2.064, -0.195),
                            (-1.978, -0.195),
                            (-1.978, -0.216),
                            (-2.064, -0.216),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.216),
                            (-1.936, -0.195),
                            (-1.85, -0.195),
                            (-1.85, -0.216),
                            (-1.936, -0.216),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.744, -0.216),
                            (-1.744, -0.195),
                            (-1.637, -0.195),
                            (-1.637, -0.216),
                            (-1.744, -0.216),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.64, -0.237),
                            (-2.64, -0.216),
                            (-2.555, -0.216),
                            (-2.555, -0.237),
                            (-2.64, -0.237),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.237),
                            (-2.299, -0.216),
                            (-2.149, -0.216),
                            (-2.149, -0.237),
                            (-2.299, -0.237),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.237),
                            (-2.064, -0.216),
                            (-1.978, -0.216),
                            (-1.978, -0.237),
                            (-2.064, -0.237),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.237),
                            (-1.936, -0.216),
                            (-1.85, -0.216),
                            (-1.85, -0.237),
                            (-1.936, -0.237),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.722, -0.237),
                            (-1.722, -0.216),
                            (-1.637, -0.216),
                            (-1.637, -0.237),
                            (-1.722, -0.237),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.64, -0.259),
                            (-2.64, -0.237),
                            (-2.576, -0.237),
                            (-2.576, -0.259),
                            (-2.64, -0.259),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.259),
                            (-2.299, -0.237),
                            (-2.128, -0.237),
                            (-2.128, -0.259),
                            (-2.299, -0.259),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.259),
                            (-2.064, -0.237),
                            (-1.978, -0.237),
                            (-1.978, -0.259),
                            (-2.064, -0.259),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.259),
                            (-1.936, -0.237),
                            (-1.85, -0.237),
                            (-1.85, -0.259),
                            (-1.936, -0.259),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.259),
                            (-1.701, -0.237),
                            (-1.637, -0.237),
                            (-1.637, -0.259),
                            (-1.701, -0.259),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.662, -0.28),
                            (-2.662, -0.259),
                            (-2.576, -0.259),
                            (-2.576, -0.28),
                            (-2.662, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.28),
                            (-2.299, -0.259),
                            (-2.213, -0.259),
                            (-2.213, -0.28),
                            (-2.299, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.192, -0.28),
                            (-2.192, -0.259),
                            (-2.107, -0.259),
                            (-2.107, -0.28),
                            (-2.192, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.28),
                            (-2.064, -0.259),
                            (-1.978, -0.259),
                            (-1.978, -0.28),
                            (-2.064, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.28),
                            (-1.936, -0.259),
                            (-1.85, -0.259),
                            (-1.85, -0.28),
                            (-1.936, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.28),
                            (-1.701, -0.259),
                            (-1.615, -0.259),
                            (-1.615, -0.28),
                            (-1.701, -0.28),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.662, -0.301),
                            (-2.662, -0.28),
                            (-2.576, -0.28),
                            (-2.576, -0.301),
                            (-2.662, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.301),
                            (-2.299, -0.28),
                            (-2.213, -0.28),
                            (-2.213, -0.301),
                            (-2.299, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.171, -0.301),
                            (-2.171, -0.28),
                            (-2.107, -0.28),
                            (-2.107, -0.301),
                            (-2.171, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.301),
                            (-2.064, -0.28),
                            (-1.978, -0.28),
                            (-1.978, -0.301),
                            (-2.064, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.301),
                            (-1.936, -0.28),
                            (-1.85, -0.28),
                            (-1.85, -0.301),
                            (-1.936, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.301),
                            (-1.701, -0.28),
                            (-1.615, -0.28),
                            (-1.615, -0.301),
                            (-1.701, -0.301),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.662, -0.323),
                            (-2.662, -0.301),
                            (-2.576, -0.301),
                            (-2.576, -0.323),
                            (-2.662, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.491, -0.323),
                            (-2.491, -0.301),
                            (-2.342, -0.301),
                            (-2.342, -0.323),
                            (-2.491, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.323),
                            (-2.299, -0.301),
                            (-2.213, -0.301),
                            (-2.213, -0.323),
                            (-2.299, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.171, -0.323),
                            (-2.171, -0.301),
                            (-2.085, -0.301),
                            (-2.085, -0.323),
                            (-2.171, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.323),
                            (-2.064, -0.301),
                            (-1.978, -0.301),
                            (-1.978, -0.323),
                            (-2.064, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.323),
                            (-1.936, -0.301),
                            (-1.85, -0.301),
                            (-1.85, -0.323),
                            (-1.936, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.323),
                            (-1.701, -0.301),
                            (-1.615, -0.301),
                            (-1.615, -0.323),
                            (-1.701, -0.323),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.662, -0.344),
                            (-2.662, -0.323),
                            (-2.576, -0.323),
                            (-2.576, -0.344),
                            (-2.662, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.491, -0.344),
                            (-2.491, -0.323),
                            (-2.342, -0.323),
                            (-2.342, -0.344),
                            (-2.491, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.344),
                            (-2.299, -0.323),
                            (-2.213, -0.323),
                            (-2.213, -0.344),
                            (-2.299, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.149, -0.344),
                            (-2.149, -0.323),
                            (-1.978, -0.323),
                            (-1.978, -0.344),
                            (-2.149, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.344),
                            (-1.936, -0.323),
                            (-1.85, -0.323),
                            (-1.85, -0.344),
                            (-1.936, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.344),
                            (-1.701, -0.323),
                            (-1.615, -0.323),
                            (-1.615, -0.344),
                            (-1.701, -0.344),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.64, -0.365),
                            (-2.64, -0.344),
                            (-2.576, -0.344),
                            (-2.576, -0.365),
                            (-2.64, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.427, -0.365),
                            (-2.427, -0.344),
                            (-2.342, -0.344),
                            (-2.342, -0.365),
                            (-2.427, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.365),
                            (-2.299, -0.344),
                            (-2.213, -0.344),
                            (-2.213, -0.365),
                            (-2.299, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.149, -0.365),
                            (-2.149, -0.344),
                            (-1.978, -0.344),
                            (-1.978, -0.365),
                            (-2.149, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.365),
                            (-1.936, -0.344),
                            (-1.85, -0.344),
                            (-1.85, -0.365),
                            (-1.936, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.701, -0.365),
                            (-1.701, -0.344),
                            (-1.637, -0.344),
                            (-1.637, -0.365),
                            (-1.701, -0.365),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.64, -0.387),
                            (-2.64, -0.365),
                            (-2.555, -0.365),
                            (-2.555, -0.387),
                            (-2.64, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.427, -0.387),
                            (-2.427, -0.365),
                            (-2.342, -0.365),
                            (-2.342, -0.387),
                            (-2.427, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.387),
                            (-2.299, -0.365),
                            (-2.213, -0.365),
                            (-2.213, -0.387),
                            (-2.299, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.128, -0.387),
                            (-2.128, -0.365),
                            (-1.978, -0.365),
                            (-1.978, -0.387),
                            (-2.128, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.387),
                            (-1.936, -0.365),
                            (-1.85, -0.365),
                            (-1.85, -0.387),
                            (-1.936, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.722, -0.387),
                            (-1.722, -0.365),
                            (-1.637, -0.365),
                            (-1.637, -0.387),
                            (-1.722, -0.387),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.64, -0.408),
                            (-2.64, -0.387),
                            (-2.555, -0.387),
                            (-2.555, -0.408),
                            (-2.64, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.427, -0.408),
                            (-2.427, -0.387),
                            (-2.342, -0.387),
                            (-2.342, -0.408),
                            (-2.427, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.408),
                            (-2.299, -0.387),
                            (-2.213, -0.387),
                            (-2.213, -0.408),
                            (-2.299, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.107, -0.408),
                            (-2.107, -0.387),
                            (-1.978, -0.387),
                            (-1.978, -0.408),
                            (-2.107, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.408),
                            (-1.936, -0.387),
                            (-1.85, -0.387),
                            (-1.85, -0.408),
                            (-1.936, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.744, -0.408),
                            (-1.744, -0.387),
                            (-1.658, -0.387),
                            (-1.658, -0.408),
                            (-1.744, -0.408),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.619, -0.43),
                            (-2.619, -0.408),
                            (-2.512, -0.408),
                            (-2.512, -0.43),
                            (-2.619, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.427, -0.43),
                            (-2.427, -0.408),
                            (-2.342, -0.408),
                            (-2.342, -0.43),
                            (-2.427, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.43),
                            (-2.299, -0.408),
                            (-2.213, -0.408),
                            (-2.213, -0.43),
                            (-2.299, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.107, -0.43),
                            (-2.107, -0.408),
                            (-1.978, -0.408),
                            (-1.978, -0.43),
                            (-2.107, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.43),
                            (-1.936, -0.408),
                            (-1.85, -0.408),
                            (-1.85, -0.85),
                            (-1.936, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.765, -0.43),
                            (-1.765, -0.408),
                            (-1.658, -0.408),
                            (-1.658, -0.43),
                            (-1.765, -0.43),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.598, -0.451),
                            (-2.598, -0.43),
                            (-2.342, -0.43),
                            (-2.342, -0.451),
                            (-2.598, -0.451),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.451),
                            (-2.299, -0.43),
                            (-2.213, -0.43),
                            (-2.213, -0.451),
                            (-2.299, -0.451),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.085, -0.451),
                            (-2.085, -0.43),
                            (-1.978, -0.43),
                            (-1.978, -0.451),
                            (-2.085, -0.451),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.451),
                            (-1.936, -0.43),
                            (-1.701, -0.43),
                            (-1.701, -0.451),
                            (-1.936, -0.451),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.555, -0.472),
                            (-2.555, -0.451),
                            (-2.384, -0.451),
                            (-2.384, -0.472),
                            (-2.555, -0.472),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.299, -0.472),
                            (-2.299, -0.451),
                            (-2.213, -0.451),
                            (-2.213, -0.472),
                            (-2.299, -0.472),
                        ]
                    ),
                    Polygon(
                        [
                            (-2.064, -0.472),
                            (-2.064, -0.451),
                            (-1.978, -0.451),
                            (-1.978, -0.472),
                            (-2.064, -0.472),
                        ]
                    ),
                    Polygon(
                        [
                            (-1.936, -0.472),
                            (-1.936, -0.451),
                            (-1.744, -0.451),
                            (-1.744, -0.472),
                            (-1.936, -0.472),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.496),
                            (2.277, -0.474),
                            (2.469, -0.474),
                            (2.469, -0.496),
                            (2.277, -0.496),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.496),
                            (2.149, -0.474),
                            (2.234, -0.474),
                            (2.234, -0.496),
                            (2.149, -0.496),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.496),
                            (1.914, -0.474),
                            (1.999, -0.474),
                            (1.999, -0.496),
                            (1.914, -0.496),
                        ]
                    ),
                    Polygon(
                        [
                            (1.658, -0.496),
                            (1.658, -0.474),
                            (1.829, -0.474),
                            (1.829, -0.496),
                            (1.658, -0.496),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.474),
                            (2.277, -0.453),
                            (2.512, -0.453),
                            (2.512, -0.474),
                            (2.277, -0.474),
                        ]
                    ),
                    Polygon(
                        [
                            (2.128, -0.474),
                            (2.128, -0.453),
                            (2.234, -0.453),
                            (2.234, -0.474),
                            (2.128, -0.474),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.474),
                            (1.914, -0.453),
                            (1.999, -0.453),
                            (1.999, -0.474),
                            (1.914, -0.474),
                        ]
                    ),
                    Polygon(
                        [
                            (1.615, -0.474),
                            (1.615, -0.453),
                            (1.871, -0.453),
                            (1.871, -0.474),
                            (1.615, -0.474),
                        ]
                    ),
                    Polygon(
                        [
                            (2.448, -0.453),
                            (2.448, -0.431),
                            (2.555, -0.431),
                            (2.555, -0.453),
                            (2.448, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.453),
                            (2.277, -0.431),
                            (2.363, -0.431),
                            (2.363, -0.453),
                            (2.277, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (2.106, -0.453),
                            (2.106, -0.431),
                            (2.234, -0.431),
                            (2.234, -0.453),
                            (2.106, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.453),
                            (1.914, -0.431),
                            (1.999, -0.431),
                            (1.999, -0.453),
                            (1.914, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (1.786, -0.453),
                            (1.786, -0.431),
                            (1.871, -0.431),
                            (1.871, -0.453),
                            (1.786, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (1.594, -0.453),
                            (1.594, -0.431),
                            (1.701, -0.431),
                            (1.701, -0.453),
                            (1.594, -0.453),
                        ]
                    ),
                    Polygon(
                        [
                            (2.469, -0.431),
                            (2.469, -0.41),
                            (2.555, -0.41),
                            (2.555, -0.431),
                            (2.469, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.431),
                            (2.277, -0.41),
                            (2.363, -0.41),
                            (2.363, -0.431),
                            (2.277, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (2.106, -0.431),
                            (2.106, -0.41),
                            (2.234, -0.41),
                            (2.234, -0.431),
                            (2.106, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.431),
                            (1.914, -0.41),
                            (1.999, -0.41),
                            (1.999, -0.431),
                            (1.914, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (1.786, -0.431),
                            (1.786, -0.41),
                            (1.871, -0.41),
                            (1.871, -0.431),
                            (1.786, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (1.572, -0.431),
                            (1.572, -0.41),
                            (1.658, -0.41),
                            (1.658, -0.431),
                            (1.572, -0.431),
                        ]
                    ),
                    Polygon(
                        [
                            (2.491, -0.41),
                            (2.491, -0.389),
                            (2.576, -0.389),
                            (2.576, -0.41),
                            (2.491, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.41),
                            (2.277, -0.389),
                            (2.363, -0.389),
                            (2.363, -0.41),
                            (2.277, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (2.085, -0.41),
                            (2.085, -0.389),
                            (2.234, -0.389),
                            (2.234, -0.41),
                            (2.085, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.41),
                            (1.914, -0.389),
                            (1.999, -0.389),
                            (1.999, -0.41),
                            (1.914, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (1.786, -0.41),
                            (1.786, -0.389),
                            (1.871, -0.389),
                            (1.871, -0.41),
                            (1.786, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (1.572, -0.41),
                            (1.572, -0.389),
                            (1.658, -0.389),
                            (1.658, -0.41),
                            (1.572, -0.41),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.389),
                            (2.512, -0.367),
                            (2.576, -0.367),
                            (2.576, -0.389),
                            (2.512, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.389),
                            (2.277, -0.367),
                            (2.363, -0.367),
                            (2.363, -0.389),
                            (2.277, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (2.064, -0.389),
                            (2.064, -0.367),
                            (2.234, -0.367),
                            (2.234, -0.389),
                            (2.064, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.389),
                            (1.914, -0.367),
                            (1.999, -0.367),
                            (1.999, -0.389),
                            (1.914, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (1.786, -0.389),
                            (1.786, -0.367),
                            (1.871, -0.367),
                            (1.871, -0.389),
                            (1.786, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (1.572, -0.389),
                            (1.572, -0.367),
                            (1.636, -0.367),
                            (1.636, -0.389),
                            (1.572, -0.389),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.367),
                            (2.512, -0.346),
                            (2.597, -0.346),
                            (2.597, -0.367),
                            (2.512, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.367),
                            (2.277, -0.346),
                            (2.363, -0.346),
                            (2.363, -0.367),
                            (2.277, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (2.064, -0.367),
                            (2.064, -0.346),
                            (2.234, -0.346),
                            (2.234, -0.367),
                            (2.064, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.367),
                            (1.914, -0.346),
                            (1.999, -0.346),
                            (1.999, -0.367),
                            (1.914, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (1.722, -0.367),
                            (1.722, -0.346),
                            (1.871, -0.346),
                            (1.871, -0.367),
                            (1.722, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (1.551, -0.367),
                            (1.551, -0.346),
                            (1.636, -0.346),
                            (1.636, -0.367),
                            (1.551, -0.367),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.346),
                            (2.512, -0.325),
                            (2.597, -0.325),
                            (2.597, -0.346),
                            (2.512, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.346),
                            (2.277, -0.325),
                            (2.363, -0.325),
                            (2.363, -0.346),
                            (2.277, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.346),
                            (2.149, -0.325),
                            (2.234, -0.325),
                            (2.234, -0.346),
                            (2.149, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (2.042, -0.346),
                            (2.042, -0.325),
                            (2.128, -0.325),
                            (2.128, -0.346),
                            (2.042, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.346),
                            (1.914, -0.325),
                            (1.999, -0.325),
                            (1.999, -0.346),
                            (1.914, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (1.722, -0.346),
                            (1.722, -0.325),
                            (1.871, -0.325),
                            (1.871, -0.346),
                            (1.722, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (1.551, -0.346),
                            (1.551, -0.325),
                            (1.636, -0.325),
                            (1.636, -0.346),
                            (1.551, -0.346),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.325),
                            (2.512, -0.303),
                            (2.597, -0.303),
                            (2.597, -0.325),
                            (2.512, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.325),
                            (2.277, -0.303),
                            (2.363, -0.303),
                            (2.363, -0.325),
                            (2.277, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.325),
                            (2.149, -0.303),
                            (2.234, -0.303),
                            (2.234, -0.325),
                            (2.149, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (2.042, -0.325),
                            (2.042, -0.303),
                            (2.106, -0.303),
                            (2.106, -0.325),
                            (2.042, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.325),
                            (1.914, -0.303),
                            (1.999, -0.303),
                            (1.999, -0.325),
                            (1.914, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (1.551, -0.325),
                            (1.551, -0.303),
                            (1.636, -0.303),
                            (1.636, -0.325),
                            (1.551, -0.325),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.303),
                            (2.512, -0.282),
                            (2.597, -0.282),
                            (2.597, -0.303),
                            (2.512, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.303),
                            (2.277, -0.282),
                            (2.363, -0.282),
                            (2.363, -0.303),
                            (2.277, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.303),
                            (2.149, -0.282),
                            (2.234, -0.282),
                            (2.234, -0.303),
                            (2.149, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (2.021, -0.303),
                            (2.021, -0.282),
                            (2.106, -0.282),
                            (2.106, -0.303),
                            (2.021, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.303),
                            (1.914, -0.282),
                            (1.999, -0.282),
                            (1.999, -0.303),
                            (1.914, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (1.551, -0.303),
                            (1.551, -0.282),
                            (1.636, -0.282),
                            (1.636, -0.303),
                            (1.551, -0.303),
                        ]
                    ),
                    Polygon(
                        [
                            (2.512, -0.282),
                            (2.512, -0.261),
                            (2.576, -0.261),
                            (2.576, -0.282),
                            (2.512, -0.282),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.282),
                            (2.277, -0.261),
                            (2.363, -0.261),
                            (2.363, -0.282),
                            (2.277, -0.282),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.282),
                            (2.149, -0.261),
                            (2.234, -0.261),
                            (2.234, -0.282),
                            (2.149, -0.282),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.282),
                            (1.914, -0.261),
                            (2.085, -0.261),
                            (2.085, -0.282),
                            (1.914, -0.282),
                        ]
                    ),
                    Polygon(
                        [
                            (1.572, -0.282),
                            (1.572, -0.261),
                            (1.636, -0.261),
                            (1.636, -0.282),
                            (1.572, -0.282),
                        ]
                    ),
                    Polygon(
                        [
                            (2.491, -0.261),
                            (2.491, -0.239),
                            (2.576, -0.239),
                            (2.576, -0.261),
                            (2.491, -0.261),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.261),
                            (2.277, -0.239),
                            (2.363, -0.239),
                            (2.363, -0.261),
                            (2.277, -0.261),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.261),
                            (2.149, -0.239),
                            (2.234, -0.239),
                            (2.234, -0.261),
                            (2.149, -0.261),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.261),
                            (1.914, -0.239),
                            (2.064, -0.239),
                            (2.064, -0.261),
                            (1.914, -0.261),
                        ]
                    ),
                    Polygon(
                        [
                            (1.572, -0.261),
                            (1.572, -0.239),
                            (1.658, -0.239),
                            (1.658, -0.261),
                            (1.572, -0.261),
                        ]
                    ),
                    Polygon(
                        [
                            (2.469, -0.239),
                            (2.469, -0.218),
                            (2.576, -0.218),
                            (2.576, -0.239),
                            (2.469, -0.239),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.239),
                            (2.277, -0.218),
                            (2.363, -0.218),
                            (2.363, -0.239),
                            (2.277, -0.239),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.239),
                            (2.149, -0.218),
                            (2.234, -0.218),
                            (2.234, -0.239),
                            (2.149, -0.239),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.239),
                            (1.914, -0.218),
                            (2.064, -0.218),
                            (2.064, -0.239),
                            (1.914, -0.239),
                        ]
                    ),
                    Polygon(
                        [
                            (1.594, -0.239),
                            (1.594, -0.218),
                            (1.679, -0.218),
                            (1.679, -0.239),
                            (1.594, -0.239),
                        ]
                    ),
                    Polygon(
                        [
                            (2.448, -0.218),
                            (2.448, -0.197),
                            (2.555, -0.197),
                            (2.555, -0.218),
                            (2.448, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.218),
                            (2.277, -0.197),
                            (2.363, -0.197),
                            (2.363, -0.218),
                            (2.277, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.218),
                            (2.149, -0.197),
                            (2.234, -0.197),
                            (2.234, -0.218),
                            (2.149, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.218),
                            (1.914, -0.197),
                            (2.042, -0.197),
                            (2.042, -0.218),
                            (1.914, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (1.786, -0.218),
                            (1.786, -0.197),
                            (1.85, -0.197),
                            (1.85, -0.218),
                            (1.786, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (1.594, -0.218),
                            (1.594, -0.197),
                            (1.701, -0.197),
                            (1.701, -0.218),
                            (1.594, -0.218),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.197),
                            (2.277, -0.175),
                            (2.533, -0.175),
                            (2.533, -0.197),
                            (2.277, -0.197),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.197),
                            (2.149, -0.175),
                            (2.234, -0.175),
                            (2.234, -0.197),
                            (2.149, -0.197),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.197),
                            (1.914, -0.175),
                            (2.021, -0.175),
                            (2.021, -0.197),
                            (1.914, -0.197),
                        ]
                    ),
                    Polygon(
                        [
                            (1.636, -0.197),
                            (1.636, -0.175),
                            (1.85, -0.175),
                            (1.85, -0.197),
                            (1.636, -0.197),
                        ]
                    ),
                    Polygon(
                        [
                            (2.277, -0.175),
                            (2.277, -0.154),
                            (2.491, -0.154),
                            (2.491, -0.175),
                            (2.277, -0.175),
                        ]
                    ),
                    Polygon(
                        [
                            (2.149, -0.175),
                            (2.149, -0.154),
                            (2.234, -0.154),
                            (2.234, -0.175),
                            (2.149, -0.175),
                        ]
                    ),
                    Polygon(
                        [
                            (1.914, -0.175),
                            (1.914, -0.154),
                            (2.021, -0.154),
                            (2.021, -0.175),
                            (1.914, -0.175),
                        ]
                    ),
                    Polygon(
                        [
                            (1.658, -0.175),
                            (1.658, -0.154),
                            (1.829, -0.154),
                            (1.829, -0.175),
                            (1.658, -0.175),
                        ]
                    ),
                ]
            ]

        mappings = [
            PadMapping(
                {
                    GND[0]: landpattern.p[1],
                    GND[1]: landpattern.p[3],
                    SIG: landpattern.p[2],
                }
            ),
            SymbolMapping(
                {
                    GND[0]: symbol.GND[0],
                    GND[1]: symbol.GND[1],
                    SIG: symbol.SIG,
                }
            ),
        ]

    sig = Port()
    gnd = Port()

    C_R = Capacitor(capacitance=0.5e-12, temperature_coefficient_code="C0G")
    L = Inductor(
        inductance=4.7e-9,
        self_resonant_frequency=Interval(7e6),
        quality_factor=Interval(8),
    )

    C_R.schematic_x_out = True
    C_R.in_bom = False
    C_R.soldered = False
    L.schematic_x_out = True
    L.in_bom = False
    L.soldered = False

    def __init__(self):
        self.nets = [
            self.sig + self.component.SIG,
            self.sig + self.L.p1,
            BridgingPinModel(self.L.p1, self.L.p2, delay=0, loss=0),
            self.L.p2 + self.C_R.p1 + self.component.SIG,
            self.gnd + self.C_R.p2 + self.component.GND[0] + self.component.GND[1],
        ]


Device: type[U_FL_R_SMT_10] = U_FL_R_SMT_10
