from __future__ import annotations

"""
Provides tools for managing a local database of financial data.

Key Features:
    - Creates and initializes a SQLite database with financial data.
    - Populates the database from various sources, including files, WRDS, and custom scripts.
    - Applies SQL cleaning operations to ensure data consistency.
    - Handles user prompts for confirmation of sensitive actions.

Main Classes:
    - `LocalDatabase`: Represents the database and offers methods for interaction.

Example Usage:
```python
from LocalDatabase import LocalDatabase

# Initialize the database
db = LocalDatabase('my_database_dir', 'database_name')

# Use the database to query data, etc.
data = db.queryDB(db.DBC.TABLE)
"""

# MIT License
# 
# Copyright (c) 2023 Andrew Maurice Perry
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
 
# standard python imports
import io
import re
import os
import sys
import zlib
import time
import numpy
import pandas
import typing
import shutil
import polars
import sqlite3
import pathlib
import datetime
import warnings
import itertools
import sqlalchemy
import connectorx
import subprocess
import importlib.util


class LocalDatabase:

    def __init__(self, 
                 save_directory: pathlib.Path | str,
                 update_all_tables: bool = False, 
                 update_created_tables: bool = False,
                 tables_to_update: list[str] | None = None, 
                 add_to_DBC_file: bool = True
                ):
        """
        Initializes a LocalDatabase object, responsible for managing a database with financial data.

        Key Responsibilities:
            - Creates the database file and directory structure if they don't exist.
            - Manages the SQLite3 connections and cursors for database interaction.
            - Populates the database with data from various sources, including:
                - Custom scripts in the 'Database/ExtraScripts' directory
                - Files in the 'Database/FILEStoDB' directory
                - WRDS (requires a valid WRDS username)
                - Custom scripts in the 'Database/CreateTables' directory
            - Applies SQL cleaning operations to tables as specified in the DatabaseContents.py file.
            - Handles user prompts for confirmation of actions, such as deleting existing tables.

        Arguments:
            - save_directory (pathlib.Path | str): Path to the root directory for the database.
            - update_all_tables (bool, optional): If True, deletes and recreates the entire database.
            - update_created_tables (bool, optional): If True, deletes and recreates tables generated by custom scripts.
            - tables_to_update (list, optional): List of specific tables to update.

        Raises:
            - _config.Exceptions.BuildError: If errors occur during database initialization.
            - _config.Exceptions.CancelOperation: If the user cancels an operation.

        Example:
        ```python
        from LocalDatabase import LocalDatabase

        db = LocalDatabase('my_database_dir', 'database_name')
        ```
        """    
        self._path_to_this_file = pathlib.Path(__file__).parent.resolve()
        # add this files directory to path
        sys.path.append(str(self._path_to_this_file))

        # project specific imports
        global _config
        global _util_funcs
        import _config
        import _util_funcs

        init_start = time.time()
            
        # convert save_directory to pathlib.Path if it is not already
        if(isinstance(save_directory, str)):
            save_directory = pathlib.Path(save_directory)
        
        self.database_name = save_directory.name
        self._root_path = save_directory.resolve()
        self._path_to_db = self._root_path / f'{self.database_name}.db'
        self._sqlite_db_path = f'sqlite:///{self._path_to_db}'
        self._path_to_base_files = self._path_to_this_file / _config.Directories._base_files
        self._path_to_shell_files = self._path_to_this_file / _config.Directories._shell_files
        self._path_to_download_files = self._path_to_this_file / _config.Directories._download_files
        self._path_to_files_to_load = self._root_path / _config.Directories.FILEStoDB
        self._path_to_create_tables = self._root_path / _config.Directories.CreateTables
        self._path_to_extra_scripts = self._root_path / _config.Directories.ExtraScripts
        self._path_to_databasecontents = self._root_path / 'DatabaseContents.py'

        # database names
        self._printpath_FILEStoDB = f'{self.database_name}/{_config.Directories.FILEStoDB}'
        self._printpath_CreateTables = f'{self.database_name}/{_config.Directories.CreateTables}'
        self._printpath_ExtraScripts = f'{self.database_name}/{_config.Directories.ExtraScripts}'

        # public class attributes
        self.today_date = datetime.datetime.now()
        self.update_all_tables = update_all_tables
        self.min_date = datetime.datetime(1850, 1, 1)
        self.update_created_tables = update_created_tables
        self.tables_to_update: list[str] = tables_to_update if tables_to_update else []
        self.db_modified: bool = False
        self.add_to_DBC_file = add_to_DBC_file
        
        def _create_blank_structure():
            for sub_dir in _config.Directories._sub_dirs:
                os.mkdir(self._root_path / sub_dir)

            # copy files to directory
            shutil.copy(self._path_to_base_files / 'DatabaseContents.py', self._root_path)
            shutil.copy(self._path_to_base_files / 'ExampleCreateTable.py', self._path_to_create_tables)
            shutil.copy(self._path_to_base_files / 'ExampleExtraScript.py', self._path_to_extra_scripts)
            
            print(_config.Messages.FIRST_BUILD.format(
                    color = _config.bcolors.WARNING, 
                    db = self._path_to_db
                )
            )

        if self._root_path.exists():
            if not any(self._root_path.iterdir()):
                _create_blank_structure()
            else:
                if not self._path_to_databasecontents.exists():
                    # Issue a runtime warning and list directory contents (including hidden files)
                    warning_msg = (
                        f"Directory '{self._root_path}' is not empty. "
                        "Contents (including hidden files):\n"
                        + "\n".join(sorted(os.listdir(self._root_path)))
                    )
                    raise RuntimeError(warning_msg)
        else:
            self._root_path.mkdir(parents=True)
            _create_blank_structure()

        # add the path to the database contents
        global DBC
        spec = importlib.util.spec_from_file_location(
            'DBC', 
            self._path_to_databasecontents
        )
        DBC = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(DBC)

        # used to call DB params from script
        self.DBC = DBC

        # must specify WRDS username if Tables.WRDS_TABLES is populated
        if(self.DBC.Tables.WRDS_TABLES):
            # wrds tables populated
            self._wrds_username = _util_funcs._rgetattr(self.DBC.Tables, 'WRDS_USERNAME')

        # sql engine for failover reading from database
        self._SQL_ENGINE = sqlalchemy.create_engine(url = self._sqlite_db_path)

        #############################################################################################
        # Remove tables from DB

        if(self.update_all_tables or self.update_created_tables or len(self.tables_to_update) > 0):
            # used to update all tables
            if(self.update_all_tables and self._path_to_db.exists()):
                response = self._get_yn_response(
                    input_message = _config.Messages.UPDATING_ALL_TABLES.format(color = _config.bcolors.WARNING)
                )
                if(response == 'y'):
                    print(_config.Messages.DELETE_DATABASE.format(color = _config.bcolors.INFO))
                    os.remove(self._path_to_db)
                    self.db_modified = True
                else:
                    raise RuntimeError(
                        _config.Messages.ABORT_INIT.format(
                            color = _config.bcolors.FAIL
                            )
                        )

            # check if any tables that are being updated are original tables
            raw_tables = self.DBC.Tables.EXTRA_TABLES + self.DBC.Tables.FILES_TABLES + self.DBC.Tables.WRDS_TABLES
            updated_og_tables = [ele for ele in self.tables_to_update if ele in raw_tables]
            if(len(updated_og_tables) != 0):
                response = self._get_yn_response(
                    input_message = _config.Messages.UPDATING_OG_TABLE.format(
                        color = _config.bcolors.WARNING
                    )
                )
                if(response == 'y'):
                    self.update_created_tables = True

            # create list of tables to delete
            tables_to_delete = self.tables_to_update
            if(self.update_created_tables):
                tables_to_delete += DBC.Tables.CREATED_TABLES

            # list of current tables
            # check to see if all required tables are present, if not load the ones that are missing
            self.curr_tables = self.get_table_names()

            # delete tables that should be updated

            # final check before delete
            if(tables_to_delete):
                response = self._get_yn_response(
                    input_message = _config.Messages.CONFIRM_DELETE.format(
                                        color = _config.bcolors.WARNING,
                                        obj = tables_to_delete
                                    )
                )
                if(response == 'n'):
                    raise RuntimeError(
                        _config.Messages.ABORT_OPERATION.format(
                            color = _config.bcolors.INFO,
                            obj = '<DELETE>'
                        )
                    )
                        
                con = sqlite3.connect(self._path_to_db)
                cur = con.cursor()
                for table_name in tables_to_delete:
                    if(table_name in self.curr_tables):
                        print(_config.Messages.DROPPING_TABLE.format(
                            color = _config.bcolors.INFO, 
                            obj = table_name
                            )
                        )
                        cur.execute(_config.SQLCommands.DROP_TABLE.format(
                            table = table_name
                            )
                        )
                        con.commit()
                        self.db_modified = True
                        self.curr_tables.remove(table_name)
                con.close()

        ##################################################################################################
        # Add tables to DB

        # reset current tables
        self.curr_tables = self.get_table_names()
        tables_at_initialization = self.curr_tables.copy()

        # Loading Tables Steps:
        #   1. Load tables that are special from 'Database/ExtraScripts'
        #   3. Read in tables from files in 'Database/FILEStoDB
        #   2. Download tables from WRDS

        ### 1. Load tables from extra scripts
        for table in self.DBC.Tables.EXTRA_TABLES:
            if(table not in self.curr_tables):
                filename: str = f'{table}.py'
                path_to_execute = self._path_to_extra_scripts / filename
                printpath = f'{self._printpath_ExtraScripts}/{filename}'
                s = time.time()
                print(_config.Messages.EXTRA_SCRIPT.format(
                        color = _config.bcolors.INFO,
                        path_to_execute = printpath
                    )
                )
                p = subprocess.run(['python3', 
                                    path_to_execute, 
                                    self._path_to_db])
                if(p.returncode != 0):
                    # subprocess crashed
                    raise RuntimeError(
                        _config.Messages.CREATE_TABLE_CRASH.format(
                            color = _config.bcolors.FAIL,
                            tab = table
                        )
                    )
                print(_config.Messages.TABLE_ADDED.format(
                        color = _config.bcolors.OK,
                        time = str(round(time.time() - s, 3))
                        )
                    )
                self.db_modified = True

        # ---------------------------------------------------
                
        ### 2. Load tables from file in Database/FILEtoDB
        added_files_to_db_flag = False
        FILEStoDB_contents = _util_funcs.list_dir(self._path_to_files_to_load)
        FILEStoDB_contents = [f[:-4] for f in FILEStoDB_contents] # remove extension
        for table in self.DBC.Tables.FILES_TABLES:
            if(table not in self.curr_tables):

                if(table not in FILEStoDB_contents):
                    print(_config.Messages.NO_DATA_FILE.format(
                            color = _config.bcolors.WARNING,
                            tab = table
                        )
                    )

                added_files_to_db_flag = True

                filepath = self._path_to_files_to_load / f'{table}.csv'
                printpath = f'{self._printpath_FILEStoDB}/{table}.csv'

                s = time.time()
                print(_config.Messages.CSV_ADD_TABLE.format(
                        color = _config.bcolors.INFO,
                        tab = printpath,
                        db = self._path_to_db.name)
                    )
                try:
                    result = subprocess.run(['bash', 
                                     f'{self._path_to_shell_files}/sqlite_csv_import.sh', 
                                     f'{self._path_to_db}', 
                                     f'{filepath}',
                                     f'{table}'], check=True)
                except:
                    raise RuntimeError(
                        _config.Messages.CSV_TO_SQL_FAIL.format(
                            color = _config.bcolors.FAIL
                        )
                    )
                print(_config.Messages.TABLE_ADDED.format(
                        color = _config.bcolors.OK, 
                        time = str(round(time.time() - s, 3))
                    )
                )
                self.db_modified = True
                    
        if(added_files_to_db_flag):
            print(_config.Messages.FILES_TO_DB_SUCCESS.format(
                color = _config.bcolors.OK
                )
            )
            print(_config.Messages.COMPRESS_FILES.format(
                color = _config.bcolors.INFO, 
                path = self._printpath_FILEStoDB
                )
            )
    
        # ---------------------------------------------------
                
        ### 3. Download tables from WRDS

        # local table names for WRDS tables
        # if no WRDS username skip
        if(not self.DBC.Tables.WRDS_USERNAME is None):
            local_names_auto_download_tables = [name.replace('.', '_') for name in DBC.Tables.WRDS_TABLES]
            if(not all(elem in self.curr_tables for elem in local_names_auto_download_tables)):
                missing_tables = list(set(local_names_auto_download_tables) - set(self.curr_tables))
                str_tables = ','.join(missing_tables)
                print(_config.Messages.MISSING_TABLE.format(
                        color = _config.bcolors.INFO,
                        obj = missing_tables
                    )
                )
                if(not isinstance(DBC.Tables.WRDS_USERNAME, str)):
                    raise RuntimeError(
                        _config.Messages.MISSING_WRDS_USERNAME.format(
                            color = _config.bcolors.FAIL
                        )
                    )
                p = subprocess.run(['python3', f'{self._path_to_download_files}/wrds_download_script.py', 
                                    self._path_to_db, self._wrds_username, str_tables])
                if(p.returncode != 0):
                    # subprocess crashed
                    raise RuntimeError(
                        _config.Messages.DOWNLOAD_TABLES_CRASH.format(
                            color = _config.bcolors.FAIL
                        )
                    )
                self.db_modified = True
                print(_config.Messages.RAW_WRDS_ADDED.format(
                        color = _config.bcolors.OK
                    )
                )
        
        ###########################################################################################################
        
        # Apply SQL cleaning to all tables
        
        # get tables added and reset curr tables
        tables_added = _util_funcs.list_diff(self.get_table_names(), tables_at_initialization)
        self.curr_tables = self.get_table_names()
        
        for table in tables_added:
            # apply null to all missing tables
            print(
                _config.Messages.CLEANING_TABLE.format(
                    color = _config.bcolors.INFO, 
                    tab = table
                )
            )
            s = time.time()
            result = subprocess.run(['bash', 
                                     f'{self._path_to_shell_files}/sqlite_null_table.sh', 
                                     f'{self._path_to_db}', 
                                     f'{table}'],
                                capture_output = True,
                                text = True
                                )

            if(result.returncode != 0):
                # subprocess crashed
                raise RuntimeError(
                    _config.Messages.SQLITE_NULL_TABLE_CRASH.format(
                        color = _config.bcolors.WARNING,
                    )
                )
            
            # apply special cleaning to other tables
            con = sqlite3.connect(self._path_to_db)
            cur = con.cursor()
            if(table in DBC.Tables.SQL_CLEANING):
                ops_dict = DBC.Tables.SQL_CLEANING[table]
                for (op, cols) in ops_dict.items():
                    if(op not in _config.SQLCommands.SQL_DICT):
                        raise RuntimeError(
                            _config.Messages.INVALID_SQL_CLEANING_OPERATION.format(
                                    color = _config.bcolors.FAIL,
                                    tab = table
                                )
                            )
                    sql_command = _config.SQLCommands.SQL_DICT[op]
                    for col in cols:
                        cur.execute(sql_command.format(table = table, col = col))
                        con.commit()
            con.close()
            print(_config.Messages.TABLE_CLEANED.format(
                color = _config.bcolors.OK, 
                time = str(round(time.time() - s, 3))
                )
            )
            
        ###########################################################################################################
        # Apply SQL Type inference
        
        for table in tables_added:
            # apply null to all missing tables
            print(
                _config.Messages.SQLITE_TYPE_INFERRING.format(
                    color = _config.bcolors.INFO, 
                    tab = table
                )
            )
            s = time.time()
            result = subprocess.run(['bash', 
                                     f'{self._path_to_shell_files}/sqlite_type_infer.sh', 
                                     f'{self._path_to_db}', 
                                     f'{table}'],
                                capture_output = True,
                                text = True
                                )

            if(result.returncode != 0):
                # subprocess crashed
                print(_config.Messages.SQLITE_TYPE_INFER_CRASH.format(
                        color = _config.bcolors.WARNING
                    )
                )
                continue
            
            # parse the type‐inference output, e.g. "PERMNO:INTEGER,date:INTEGER"
            type_info_str = result.stdout.strip()
            type_info = {}
            for segment in type_info_str.split(','):
                if ':' not in segment:
                    continue
                col, dtype = segment.split(':', 1)
                type_info[col] = dtype
            
            # create a new table with the inferred types                
            select_items: list[str] = []

            for col, affinity in type_info.items():
                if affinity.upper() == "TEXT":
                    # keep text columns as‑is
                    select_items.append(f'    "{col}"')
                else:
                    select_items.append(
                f'    CAST("{col}" AS {affinity.upper()}) AS "{col}"'
            )

            select_clause = ",\n".join(select_items)
            
            untyped_table = f'{table}_untyped'
            typed_table = table
            
            con = sqlite3.connect(self._path_to_db)
            cur = con.cursor()

            cur.execute(_config.SQLCommands.RENAME_TABLE.format(  
                old = table,  
                new = untyped_table  
            ))  
            cur.execute(_config.SQLCommands.CAST_TABLE.format(  
                new = typed_table,  
                cast = select_clause,  
                old = untyped_table  
            ))  
            cur.execute(_config.SQLCommands.DROP_TABLE.format(  
                table = untyped_table  
            ))  
            con.commit()  

            con.close()
            print(_config.Messages.TABLE_TYPE_INFERRED.format(
                color = _config.bcolors.OK, 
                tab = table,
                time = str(round(time.time() - s, 3))
                )
            )
            
        ###########################################################################################################
        # Add created tables
                        
        for table in DBC.Tables.CREATED_TABLES:
            if(table not in self.curr_tables):
                filename = f'{table}.py'
                path_to_execute = self._path_to_create_tables / filename
                printpath = f'{self._printpath_CreateTables}/{filename}' 
                s = time.time()
                print(_config.Messages.BUILDING_TABLE.format(
                        color = _config.bcolors.INFO,
                        tab = table,
                        file = printpath
                    )
                )
                p = subprocess.run(['python3', 
                                    path_to_execute, 
                                    self._path_to_db])
                if(p.returncode != 0):
                    # subprocess crashed
                    raise RuntimeError(
                        _config.Messages.CREATE_TABLE_CRASH.format(
                                color = _config.bcolors.FAIL,
                                tab = table
                            )
                        )
                print(_config.Messages.TABLE_ADDED.format(
                    color = _config.bcolors.OK,
                    time = str(round(time.time() - s, 3))
                    )
                )
                self.db_modified = True

        ########################################################################################################
        # Add to the end of the proposed DB params file
        
        # rest current tables
        # get tables added and reset curr tables
        tables_added = _util_funcs.list_diff(self.get_table_names(), tables_at_initialization)

        # establish connection to database
        if(tables_added):
            if(self.add_to_DBC_file):
                response = 'y'
            else:
                response = self._get_yn_response(
                    input_message = _config.Messages.ADD_TO_DBC.format(
                        color = _config.bcolors.INFO,
                        tables = tables_added
                    )
                )
            
            if(response == 'y'):
                
                con = sqlite3.connect(self._path_to_db)
                cur = con.cursor()
    
                with open(self._path_to_databasecontents, 'a') as write_DBC:
                    for table in tables_added:
                    
                        # extract info from sql
                        table_info = self._extract_pragma_table_info(cur = cur, table = table)
                        column_names = table_info['names']
                        column_names = [f'\'{n}\'' for n in column_names]
                        column_dtypes = table_info['dtypes']
                        vars_dtypes = dict(map(lambda i, j: (i, j), column_names, column_dtypes))
    
                        # convert dictionary into list of strings 'k: v'
                        _tmp = [None] * len(vars_dtypes.keys())
                        for i, (k, v) in enumerate(vars_dtypes.items()):
                            _tmp[i] = f'{k.lower()}: {v}'
    
                        # create string to be written to class
                        PYTHON_PAD = ' ' * 4
                        class_string = '\n'
                        class_string += f'class {table}:\n\n'
                        class_string += f'{PYTHON_PAD}TABLE = \'{table}\'\n\n'
                        class_string += f'{PYTHON_PAD}DEFAULT_ID = None\n'
                        class_string += f'{PYTHON_PAD}DEFAULT_DATE = None\n\n'
                        class_string += f'{PYTHON_PAD}DEFAULT_VARS = []\n\n'
                        class_string += f'{PYTHON_PAD}VARS_DATA_TYPE = '
                        class_string += '{\n'
                        class_string += self._format_vars_dtype_DBC(_tmp, max_length = 110, 
                                                                    initial_pad = ' ' * 22)
                        class_string += '\n' + ' ' * 21
                        class_string += '}\n\n'
    
                        write_DBC.write(class_string)
    
                write_DBC.close()
                con.close() # i hate myself
                
                
        # reset spec after contents written to DBC
        spec = importlib.util.spec_from_file_location(
            'DBC', 
            self._path_to_databasecontents
        )
        DBC = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(DBC)

        # used to call DB params from script
        self.DBC = DBC
                
        # build dictionary to search after contents have been added to the DBC file
        # used to search DBC params
        DBC_dict = self._db_params_to_dict(DBC)        
        [flat_dict] = pandas.json_normalize(DBC_dict, sep = '.').to_dict(orient = 'records')
        self._DBC_flat = {}
        for (k, v) in flat_dict.items():
            k = k.replace('.subclasses', '')
            k = k.replace('.attr', '')
            self._DBC_flat[k] = v
            
        ##########################################################################################
        # Download SIC Classification Data
        
        # update curr tables
        self.curr_tables = self.get_table_names()
        
        if('FFSIC' not in self.curr_tables):
            # sic table not present, download it
            print(_config.Messages.DOWNLOAD_SIC.format(
                    color = _config.bcolors.INFO
                )
            )
            s = time.time()
            p = subprocess.run(['python3', f'{self._path_to_download_files}/ffsic_download_script.py', 
                                self._path_to_db])
            if(p.returncode != 0):
                # subprocess crashed
                raise RuntimeError(
                    _config.Messages.DOWNLOAD_SIC_CRASH.format(
                        color = _config.bcolors.FAIL
                    )
                )
            print(_config.Messages.SIC_ADDED.format(
                    color = _config.bcolors.OK
                )
            )
            self.db_modified = True
        
        ##########################################################################################
        # Initialized!

        init_end = time.time()
        if(self.db_modified):
            print(_config.Messages.DATABASE_INITIALIZED.format(
                    color = _config.bcolors.OK,
                    time = str(datetime.timedelta(seconds = init_end - init_start))
                )
            )
 
    def __str__(self) -> str:
        """
        Return a formatted string representation of the database and its tables.

        Returns:
            str: A formatted string containing information about the database and its tables.
        """
        NUM_COLS = 8
        res = f'{_config.bcolors.HEADER}Database Name: {self._path_to_db}{_config.bcolors.ENDC}\n'
        con = sqlite3.connect(self._path_to_db)
        cur = con.cursor()
        tablenames = cur.execute("""SELECT name FROM sqlite_master WHERE type = 'table'""")
        for name_tuple in tablenames.fetchall():
            name = name_tuple[0]
            res += f'{_config.bcolors.UNDERLINE}Table name: {name}{_config.bcolors.ENDC}\n'
            cols_cur = cur.execute(f"""SELECT * FROM {name} LIMIT 1""")
            cols = list(map(lambda x: x[0], cols_cur.description))
            cols = [col.lower().strip() for col in cols]
            res += self._format_columns_repr(cols, NUM_COLS)
            res += '\n'
        con.close()
        return(res)
    
    def df_to_db(self, 
                 df: typing.Union[pandas.DataFrame, polars.DataFrame],
                 table_name: str
                ) -> None:
        """
        Save a DataFrame to a database table.

        Parameters
        ----------
        df : typing.Union[pandas.DataFrame, polars.DataFrame]
            The input DataFrame to save to the database. Supports 
            pandas and polars DataFrames.

        table_name : str
            The name of the table to which the DataFrame will be 
            saved. If the table exists, it will be replaced.

        Raises
        ------
        TypeError
            If the input DataFrame is not a pandas or polars DataFrame.

        Examples
        --------
        >>> import pandas as pd
        >>> df = pd.DataFrame({'A': [1, 2, 3]})
        >>> DB = FinToolsAP.LocalDatabase(save_directory = ...,
                                          database_name = ...)
        >>> DB.df_to_db(df = df, table_name = name)
        """
    
        is_pandas = isinstance(df, pandas.DataFrame)
        is_polars = isinstance(df, polars.DataFrame)
        
        if(is_pandas):
            df.to_sql(name = table_name, 
                      con = self._SQL_ENGINE, 
                      if_exists = 'replace', 
                      index = False
                    )
        elif(is_polars):
            df.write_database(table_name = table_name,
                              connection = self._sqlite_db_path,
                              if_table_exists = 'replace'
                            )
        else:
            raise TypeError(
                f'df: expected type pandas.DataFrame or polars.DataFrame, got {type(df).__name__!r}'
                )
    
    def get_table_names(self):
        """
        Retrieve the names of all tables in the 
            connected SQLite database.

        Returns
        -------
        list
            A list of table names in the database.
        """
        con = sqlite3.connect(self._path_to_db)
        cur = con.cursor()
        tablenames = cur.execute(_config.SQLCommands.GET_TABLE_NAMES)
        res = [tab[0] for tab in tablenames.fetchall()]
        con.close()
        return(res)

    def decompress_csv(input_path: pathlib.Path | str, 
                       output_path: pathlib.Path | str, 
                       tables: list[str] = None) -> None:
        """Decompresses CSV files from a specified input directory and copies them to an output directory.

        This function iterates through files in the input directory, decompresses CSV files with a '.zip'
        extension, and copies the decompressed files to the output directory. It performs input validation,
        handles potential errors, and provides informative messages.

        Args:
            input_path: The path to the directory containing the compressed CSV files.
            output_path: The path to the directory where the decompressed CSV files should be copied.
            tables: An optional list of specific table names (without the '.zip' extension) to decompress.
                    Defaults to None, which decompresses all CSV files in the input directory.

        Raises:
            TypeError: If input_path or output_path is not of type str or pathlib.Path.
            FileNotFoundError: If input_path or output_path does not exist.
            NotADirectoryError: If input_path or output_path is not a directory.
            TypeError: If tables is not a list or None.
            Exception: If decompression fails for a specific file.

        Returns:
            None
        """

        if(isinstance(input_path, str)):
            input_path = pathlib.Path(input_path)
        elif(isinstance(input_path, pathlib.Path)):
            pass
        else:
            raise TypeError(_config.Messages.PATH_FORMAT.format(color = _config.bcolors.FAIL, 
                                                                obj = 'input_path'))
        
        if(isinstance(output_path, str)):
            output_path = pathlib.Path(output_path)
        elif(isinstance(output_path, pathlib.Path)):
            pass
        else:
            raise TypeError(_config.Messages.PATH_FORMAT.format(color = _config.bcolors.FAIL, 
                                                                obj = 'output_path'))

        if(not os.path.exists(input_path)):
            raise FileNotFoundError(_config.Messages.NOT_FOUND.format(color = _config.bcolors.FAIL, 
                                                                      obj = 'input_path'))
        
        if(not os.path.exists(output_path)):
            raise FileNotFoundError(_config.Messages.NOT_FOUND.format(color = _config.bcolors.FAIL, 
                                                                      obj = 'output_path'))
        
        if(not os.path.isdir(input_path)):
            raise NotADirectoryError(_config.Messages.MUST_BE_DIRECTORY.format(color = _config.bcolors.FAIL, 
                                                                               obj = 'input_path'))
        
        if(not os.path.isdir(output_path)):
            raise NotADirectoryError(_config.Messages.MUST_BE_DIRECTORY.format(color = _config.bcolors.FAIL, 
                                                                               obj = 'output_path'))
        
        if(not (isinstance(tables, list) or tables is None)):
            raise TypeError(_config.Messages.TABLES_TO_DECOMPRESS.format(color = _config.bcolors.FAIL,
                                                                         obj = 'tables'))
        
        if(tables is None):
            all_files = True

        dir_content = os.listdir(input_path)
        for file in dir_content:
            if(not file.endswith('.zip')): continue
            if(not all_files):
                if(not file.strip('.zip') in tables): continue 

            compressed_path = input_path / f'{file}'
            uncompressed_path = input_path / f'{file.strip(".zip")}.csv'

            try:
                # compress csv files
                with open(compressed_path, mode = 'rb') as file_in:
                    with open(uncompressed_path, mode = 'wb') as file_out:
                        data = file_in.read()
                        uncompressed_data = zlib.decompress(data)
                        file_out.write(uncompressed_data)
                        file_in.close()
                        file_out.close()
                shutil.copy(uncompressed_path, output_path)
                os.remove(compressed_path)
                os.remove(uncompressed_path)
            except Exception:
                print(_config.Messages.DECOMPRESS_FAIL.format(color = _config.bcolors.WARNING,
                                                              file = compressed_path))
                continue
    

    def raw_sql(self, sql_str):
        """Executes raw SQL and returns the results as a list of tuples, but only after
        explicit user confirmation to acknowledge potential security risks.
    
        This function prompts the user to confirm their understanding of the potential
        security implications of executing raw SQL. If the user confirms, the provided
        SQL string is executed, and the results are returned as a DataFrame. If the user
        declines, the operation is aborted, and None is returned.
    
        Args:
            sql_str: The raw SQL string to execute.
    
        Returns:
            A pandas DataFrame containing the query results if the user confirms,
            or None if the user declines.
        """
        while(True):
            response = input(_config.Messages.COMPROMISE_DATABASE.format(
                color = _config.bcolors.WARNING
                )
            )
            if(response.strip().lower() == 'y'):
                con = sqlite3.connect(self._path_to_db)
                cur = con.cursor()
                exe = cur.execute(sql_str)
                res = exe.fetchall()
                con.commit()
                con.close()
                return(res)
            elif(response.strip().lower() == 'n'):
                print(_config.Messages.ABORT_OPERATION.format(
                        color = _config.bcolors.INFO,
                        obj = '<raw_sql>'
                    )
                )
                return(None)
            else:
                continue
        
    def queryDB(self, 
                table_info: typing.Union[typing.Any, str], 
                return_type: str = 'pandas',
                row_limit: int = None, 
                suppress: bool = False, 
                **kwargs
            ) -> pandas.DataFrame | polars.DataFrame:
        """Queries a database table and returns a pandas or polars DataFrame.

        This function handles date formatting, constructs query components based on input arguments,
        incorporates default filtering criteria, executes the query, cleans and formats the results,
        and returns a DataFrame.
    
        Args:
            table_info: An object containing information about the table to query, including
                        table name, default variables, data types, etc.
            **kwargs: Keyword arguments specifying query variables, date range, and other filtering
                       criteria. Valid keyword arguments include:
                       - 'start_date' (datetime.datetime or str in '%Y-%m-%d' format)
                       - 'end_date' (datetime.datetime or str in '%Y-%m-%d' format)
                       - 'vars' (list of variables to query)
                       - Other variables specified in the DatabaseContents class
            return_type (str, optional): The return type of the resulting data. Default is pandas.
                Options are pandas or polars.
    
        Returns:
            A pandas DataFrame containing the queried data.
    
        Raises:
            TypeError: If start_date or end_date are not of the correct type.
        """
        
        # dates 
        if('start_date' in kwargs): 
            start_date = kwargs['start_date']
            is_str:      bool = isinstance(start_date, str)
            is_datetime: bool = isinstance(start_date, datetime.datetime)
            if is_datetime or is_str:
                if is_str:
                    start_date: datetime.datetime = datetime.datetime.strptime(start_date, '%Y-%m-%d')
            else:
                raise TypeError(_config.Messages.DATE_FORMAT.format(color = _config.bcolors.FAIL,
                                                                    date = 'start_date'))
        else:
            start_date: datetime.datetime = self.min_date

        if('end_date' in kwargs): 
            end_date = kwargs['end_date']
            is_datetime = isinstance(end_date, datetime.datetime)
            is_str = isinstance(end_date, str)
            if(is_datetime or is_str):
                if(is_str):
                    end_date = datetime.datetime.strptime(end_date, '%Y-%m-%d')
            else:
                raise TypeError(_config.Messages.DATE_FORMAT.format(color = _config.bcolors.FAIL, 
                                                                    date = 'end_date'))
        else:
            end_date = self.today_date

        # table can be loaded using string 
        if(isinstance(table_info, str)):
            table_info = self._DBC_flat[f'{table_info}.class']
            
        query_components = self._build_query_components(
            table_info = table_info, 
            start_date = start_date,
            end_date = end_date,
            kwrd_dict = kwargs
        )
        
        # return type of data
        query_components['return_type'] = return_type
        query_components['suppress'] = suppress
        query_components['row_limit'] = row_limit
        
        # search for default filtering
        table_name = str(table_info).replace('<class \'DBC.', '')
        table_name = table_name.replace('<class \'DBC.', '')
        table_name = table_name.replace('\'>', '')

        reserved_attr = _config.RequiredAttributes.GENERIC_TABLE + _config.RequiredAttributes.RESERVED_ATTR
        additional_defaults = _util_funcs.list_diff(list1 = self._DBC_flat[table_name],
                                                    list2 = reserved_attr
                                                    )
        for default in additional_defaults:
            if('DEFAULT_' in default):
                default_var = default.replace('DEFAULT_', '').lower()
                query_components[default_var] = _util_funcs._rgetattr(table_info, default)
        
        raw_df = self._query_database(query_components)

        # return dataframe
        return(raw_df)
    
    
    def getFFIndustryClassification(self, sic, info: str = None, level: int = 49):
        """
        Look up the Fama-French industry assignment for a single 4-digit
        Standard Industrial Classification (SIC) code using a pre-built
        SQLite table (``FFSIC_<level>``).

        Parameters
        ----------
        sic : int | str
            Four-digit SIC code (e.g. ``6021``).  May be passed as an
            integer or a zero-padded string.  Codes outside **0100–9999**
            raise ``ValueError``.
        info : {"id", "abbrev", "group", "desc", "sic_start", "sic_end"}, optional
            *If ``None`` (default)* return **all** six fields.
            Otherwise, return only the requested field:

            * ``"id"``         – Fama-French numeric portfolio identifier  
            * ``"abbrev"`` – dataset’s ticker-style abbreviation (e.g. ``'Fin'``)  
            * ``"group"``   – textual portfolio label (e.g. ``'Finance'``)  
            * ``"desc"``    – long portfolio description  
            * ``"sic_start"`` / ``"sic_end"`` – interval bounds that
              captured the SIC code
        level : {10, 12, 17, 30, 38, 48, 49}, default 49
            Selects which Fama-French industry scheme to query
            (49-, 30-, 12-industry, etc.).  The corresponding SQLite table
            **must** exist and follow the column order  
            ``(id, abbrev, group, desc, sic_start, sic_end)``.

        Returns
        -------
        tuple | int | str
            * If *info* is ``None``:  
              ``(id, abbrev, group, desc, sic_start, sic_end)``
            * Otherwise: the single requested value.

        Raises
        ------
        TypeError
            If *sic* is not ``int`` or ``str``.
        ValueError
            If *sic* is not a valid 4-digit code, falls outside the
            retrieved interval, or *info* is not one of the allowed
            keywords.

        Notes
        -----
        * The method assumes the SQLite database path is stored in
          ``self._path_to_db`` and that the connection table is named
          exactly ``FFSIC_<level>``.
        * The table **must** contain mutually exclusive, inclusive
          intervals of SIC codes.  Overlapping intervals could cause the
          query to return multiple rows and break the logic.

        Examples
        --------
        >>> db = FFIndustryLookup("ff_49_industries.sqlite")
        >>> db.getFFIndustryClassification(6021)
        (44, 'Fin', 'Finance', 'National Commercial Banks', 6000, 6029)

        >>> db.getFFIndustryClassification("3711", info="group", level=30)
        'Consumer Durables'
        """
        if isinstance(sic, int):
            if sic < 1000 or sic > 9999:
                raise ValueError(f"sic must be a 4-digit integer, got {sic!r}")
        elif isinstance(sic, str):
            if len(sic) != 4 or not sic.isdigit():
                raise ValueError(f"sic string must consist of 4 numeric characters, got {sic!r}")
        else:
            raise TypeError(f"sic must be an int or str, got {type(sic).__name__!r}")
        
        con = sqlite3.connect(self._path_to_db)
        cur = con.cursor()        
        exe = cur.execute(f"""SELECT * FROM FFSIC_{level} WHERE {sic} BETWEEN sic_start AND sic_end""")
        res = exe.fetchall()
        con.commit()
        con.close()

        id          = res[0][0]
        abbrev      = res[0][1]
        group       = res[0][2]
        desc        = res[0][3]
        sic_start   = res[0][4]
        sic_end     = res[0][5]
        
        # validate sic is between sic_start and sic_end
        if not (sic_start <= sic <= sic_end):
            raise ValueError(f"SIC code {sic!r} is not within the range {sic_start!r} to {sic_end!r} for level {level}.")
        
        if(info is not None):
            if(info.lower() == 'id'):
                return id
            elif(info.lower() == 'abbrev'):
                return abbrev
            elif(info.lower() == 'group'):
                return group
            elif(info.lower() == 'desc'):
                return desc
            elif(info.lower() == 'sic_start'):
                return sic_start
            elif(info.lower() == 'sic_end'):
                return sic_end
            else:
                raise ValueError(f"Invalid info parameter: {info!r}. Must be one of: 'id', 'abbrev', 'group', 'desc', 'sic_start', 'sic_end'.")
        else:
            return id, abbrev, group, desc, sic_start, sic_end

# ----------------------------------------------------------------------------------------------------------------------------
# INTERNAL METHODS (class <QueryWRDS>)
#
# These are internal methods and should only be called within this class. Functionality and accuracy of these methods cannot
# guaranteed if they are called outside of this class.
# ----------------------------------------------------------------------------------------------------------------------------

    def _list_to_sql_str(self, lst: list, table: str = None) -> str:
        """Converts a list of variables to an SQL string with optional table qualification.

        Args:
            lst: A list of variable names to be included in the SQL string.
            table: The name of the table to qualify the variables with (optional). If provided,
                  each variable will be prefixed with the table name and a dot (e.g., "table.variable").

        Returns:
            An SQL string containing the variables, separated by commas and enclosed in single quotes.
            If a table name is provided, the variables will be qualified with the table name.
        """
        if(len(lst) == 1 and lst[0] == '*'):
            return('*')
        res = ''
        for var in lst:
            if(table is None):
                res += f'\'{var}\', '
            else:
                res += f'{table}.{var}, '
        res = res[:-2]
        return(res)
    
    def _build_query_components(self,
                                table_info, 
                                start_date: datetime.datetime, 
                                end_date: datetime.datetime, 
                                kwrd_dict: dict[str:list|int|str|float]
                            ) -> dict[str|list]:
        """Constructs a dictionary of components for building an SQL query.

        This function assembles essential information for constructing an SQL query,
        including table information, date range, variables to query for, and any
        additional filtering criteria. It validates input types and ensures consistency.

        Steps:

        1. Gather Core Components:
           - Creates a base dictionary with table information, start date, and end date.
        2. Format Query Variables:
           - Calls the `_format_query_vars` function to process keyword arguments and
             construct a list of valid query variables.
           - Adds the formatted query variables to the query components dictionary.
        3. Handle Additional filtering:
           - Identifies any keyword arguments not related to query variables.
           - Iterates through these additional filtering arguments:
             - Converts single values (strings, floats, or integers) into lists.
             - Validates that values are either lists or have already been converted.
             - Raises an error for invalid component types.
             - Adds the filtering arguments and their values to the query components dictionary.

        Args:
            table_info: Information about the table to query.
            start_date: The starting date for the query.
            end_date: The ending date for the query.
            kwrd_dict: A dictionary of keyword arguments specifying query variables and filtering criteria.

        Returns:
            A dictionary containing all necessary components for constructing the SQL query.

        Raises:
            _config.Exceptions.QueryingError: If invalid component types are encountered.
        """
        
        query_components = {'table_info': table_info, 
                            'start_date': start_date, 
                            'end_date': end_date}
        
        # variables to query for
        query_vars = self._format_query_vars(kwrd_dict = kwrd_dict,
                                             table_info = table_info
                                            )
        query_components['vars'] = query_vars
        
        # additional filtering
        add_filtering = _util_funcs.list_diff(list(kwrd_dict.keys()), _config.KeywordArguments.QUERY_VARS)
        for kwrd in add_filtering:
            val = kwrd_dict[kwrd]
            if(isinstance(val, str) or isinstance(val, float) or isinstance(val, int)):
                val = [val]
            elif(isinstance(val, list)):
                pass
            else:
                raise TypeError(
                        _config.Messages.INVALID_COMPONENT_TYPE.format(
                                color = _config.bcolors.FAIL,
                                obj = kwrd
                            )
                        )
            query_components[kwrd] = val

        return(query_components)   
    
    def _format_query_vars(self, 
                           kwrd_dict: dict[str:list|str], 
                           table_info
                           ) -> list[str]:
        """Formats a list of query variables based on input keyword arguments and table information.

        This function handles various keyword arguments to construct a list of variables
        to be queried from a database table. It validates the queried variables against
        the table's valid variables, prioritizes default variables and essential identifiers,
        and handles potential conflicts between keyword arguments.

        Steps:

        1. Extract Table Information:
           - Retrieves default ID, default date, default variables, and valid variables (if available) from `table_info`.
        2. Check Keyword Argument Compatibility:
           - Ensures that 'vars' is not used together with 'add_vars' or 'sub_vars'.
        3. Construct Initial Query Variables List:
           - Uses 'vars' if provided, otherwise starts with default variables.
           - Inserts default ID and default date if necessary.
        4. Apply Keyword Arguments:
           - Adds variables from 'add_vars', subtracts variables from 'sub_vars', and includes filtering variables.
           - Overrides with all variables if 'all_vars' is True.
        5. Remove Duplicates:
           - Ensures unique variables in the query list.
        6. Validate Query Variables:
           - Checks if all queried variables are present in the valid variables list (if available).

        Args:
            kwrd_dict: A dictionary containing keyword arguments specifying query variables.
            table_info: An object containing information about the table to be queried.

        Returns:
            A list of formatted query variables.

        Raises:
            _config.Exceptions.QueryingError: If incompatible keyword arguments are used or invalid variables are queried.
        """

        # get specific table info
        default_id = table_info.DEFAULT_ID
        default_date = table_info.DEFAULT_DATE
        default_vars = table_info.DEFAULT_VARS

        # if not all variables are specified then cannot error check
        valid_vars = None 
        if(_util_funcs._rhasattr(table_info, 'ALL_VARS')):
            if(table_info.ALL_VARS):
                valid_vars = list(table_info.VARS_DATA_TYPE.keys())
        
        # keywords 'additional_vars' and 'vars' cannot be used simultaneously
        if('vars' in kwrd_dict and ('add_vars' in kwrd_dict or 'sub_vars' in kwrd_dict)): 
            raise ValueError(
                _config.Messages.ADDVARS_VARS_KWRDS.format(
                    color = _config.bcolors.FAIL
                )
            )
        
        # create list of the variables being queried 
        query_vars = default_vars
        if('vars' in kwrd_dict):
            # variable arguments to query for            
            query_vars = kwrd_dict['vars']
            if(not isinstance(query_vars, list)):
                query_vars = [query_vars]

            # check if database has no must have identifier
            if(not default_id is None):
                if(default_id not in query_vars): 
                    query_vars.insert(0, default_id)

            # add date if people forgot
            if(not default_date is None):
                if(default_date not in query_vars): 
                    query_vars.insert(0, default_date)
        else:
            if('add_vars' in kwrd_dict):
                query_vars = default_vars + _util_funcs.convert_to_list(kwrd_dict['add_vars'])

            if('sub_vars' in kwrd_dict):
                sub_vars = _util_funcs.convert_to_list(kwrd_dict['sub_vars'])
                query_vars = _util_funcs.list_diff(default_vars, sub_vars)

        # make sure filtering vars are also quired for
        if(kwrd_dict):
            query_vars += _util_funcs.list_diff(
                list1 = list(kwrd_dict.keys()), 
                list2 = _config.KeywordArguments.QUERY_VARS)
        # all variables
        if('all_vars' in kwrd_dict): 
            if(kwrd_dict['all_vars']):
                query_vars = ['*']
            
        # remove duplicates
        query_vars = list(dict.fromkeys(query_vars))

        # raise if no vars
        if(not query_vars):
            raise ValueError(
                _config.Messages.NO_QUERY_VARS.format(
                    color = _config.bcolors.FAIL,
                    table = table_info.TABLE
                )
            )

        # final check of valid query vars
        if(not self._all_valid(query_vars, valid_vars) and query_vars != ['*']):
            raise ValueError(
                _config.Messages.VAR_CANNOT_BE_QUERIED.format(
                    color = _config.bcolors.FAIL,
                    obj = str(_util_funcs.list_diff(query_vars, valid_vars)),
                    tab = table_info.TABLE
                )
            )

        return(query_vars)
    
    def _all_valid(self, queried: list, valid: list | None) -> bool:
        """Checks if all elements in a queried list are present within a valid list.

        This function efficiently determines whether every element in the `queried` list
        can also be found within the `valid` list. It handles cases where no explicit
        validation is required (when `valid` is None).

        Args:
            queried: A list of elements to check for validity.
            valid: A list containing the valid elements, or None if no validation is needed.

        Returns:
            True if all elements in `queried` are found within `valid`, or if `valid` is None.
            False if any element in `queried` is not found within `valid`.
        """
        if(valid is None): return(True)
        all_valid = all(elem in valid for elem in queried)
        return(all_valid)
    
    def _query_database(self, 
                        query_components: dict[str:datetime.datetime|str|float|int|list[str|float|int]|bool]
                    ) -> pandas.DataFrame:
        """Retrieves data from a database table, performs cleaning and formatting, and returns a DataFrame.

        This function executes an SQL query based on provided query components, reads the results into a DataFrame,
        applies various cleaning and formatting steps, and returns the processed DataFrame.

        Steps:

        1. Extract Table Information:
           - Retrieves table name, default date column, variable data types, and default ID column from `query_components`.
        2. Construct SQL Query:
           - Calls the `_sql_query` function to generate the SQL query string for retrieving data.
        3. Execute Query:
           - Uses `pandas.read_sql` to execute the query against the database and read the results into a DataFrame.
        4. Clean and Format Data:
           - Converts column names to lowercase.
           - Sets DataFrame column data types according to the specified variable data types.
           - Replaces Python `None` values with NumPy `np.nan` for consistency.
           - Orders the columns of the DataFrame to the same as the originally queried variables.
           - Drops any duplicate rows.
           - Sorts the DataFrame by default ID and default date columns (if available).
           - Resets the DataFrame index.

        Args:
            query_components: A dictionary containing the necessary information for constructing the SQL query
                               (see `_sql_query` function for details).

        Returns:
            A pandas DataFrame containing the queried data, cleaned and formatted.
        """

        # get table info
        TABLE_INFO = query_components['table_info']
        VARS_DATA_TYPE = TABLE_INFO.VARS_DATA_TYPE
        DEFAULT_ID = TABLE_INFO.DEFAULT_ID
        DEFAULT_DATE = TABLE_INFO.DEFAULT_DATE

        # get data types that are being quired for
        query_vars = query_components['vars']

        # data types
        query_dtype = None
        if(query_vars == ['*']):
            query_dtype = VARS_DATA_TYPE
        else:
            query_dtype = {k:v for (k, v) in VARS_DATA_TYPE.items() if k in query_vars}

        # get return type and remove from query components
        return_type = query_components['return_type']
        del query_components['return_type']

        # get suppress
        suppress = query_components['suppress']
        del query_components['suppress']

        # check that return type is valid
        if(not return_type in ['polars', 'pandas']):
            raise ValueError(_config.Messages.INVALID_RETURN_TYPE.format(color = _config.bcolors.FAIL,
                                                                         act = return_type))
        
        # read in raw dataframe from local sql database
        try:
            # try with connectorx. might run into error
            # >>> RuntimeError: Cannot infer type from null for SQLite
            # https://github.com/sfu-db/connector-x/issues/488
            raw_df = connectorx.read_sql(conn = self._sqlite_db_path, 
                                         query = self._sql_query(query_components),
                                         return_type = return_type
                                        )
        except:
            # failover to pandas if connectorx failed
            raw_df = pandas.read_sql(sql = self._sql_query(query_components),
                                     con = self._SQL_ENGINE
                                    )
            return_type = 'pandas'
    
        # make all columns lower case
        cols = raw_df.columns
        cols = [col.lower() for col in cols]
        raw_df.columns = cols  

        # return pandas or polars
        if(return_type == 'pandas'):
            # set data types
            cast_dict = {}
            # invert dictionary
            polars_to_pandas = dict((v, k) for k, v in _config.TypeMapping.TYPE_MAP.items())
            for k, v in query_dtype.items():
                if(v in _config.TypeMapping.TYPE_MAP.keys()):
                    # specified pandas for pandas
                    cast_dict[k] = v
                else:
                    # specified pandas for polars
                    cast_dict[k] = polars_to_pandas[v]

            # final check to make sure casting columns are in the 
            # dataframe. 
            cast_dict = {k:v for k, v in cast_dict.items() if k in raw_df.columns}    
            try:
                raw_df = raw_df.astype(cast_dict)
            except Exception as e:
                pass

            # replace and python objects 'None' to np.nan
            raw_df = raw_df.fillna(value = numpy.nan)

            # reset to original variables, drop duplicates, and reset the index
            if(query_vars != ['*']):
                raw_df = raw_df[query_vars]
            raw_df = raw_df.drop_duplicates()
            sorting_dims = []
            if(DEFAULT_ID is not None):
                sorting_dims.append(DEFAULT_ID)
            if(DEFAULT_DATE is not None):
                sorting_dims.append(DEFAULT_DATE)
            
            valid_sorting_dims = [dim for dim in sorting_dims if dim in raw_df.columns]
            if valid_sorting_dims:
                raw_df = raw_df.sort_values(by=valid_sorting_dims)
            raw_df = raw_df.reset_index(drop = True)
            return(raw_df)
        elif(return_type == 'polars'):
            # set data types
            cast_dict = {}
            for k, v in query_dtype.items():
                if(v in _config.TypeMapping.TYPE_MAP.values()):
                    cast_dict[k] = v
                else:
                    cast_dict[k] = _config.TypeMapping.TYPE_MAP[v]

            # final check to make sure casting columns are in the 
            # dataframe. 
            cast_dict = {k:v for k, v in cast_dict.items() if k in raw_df.columns}   

            try:
                raw_df = raw_df.cast(cast_dict)
            except:
                if(not suppress):
                    warnings.warn(
                        message = _config.Messages.COULD_NOT_CAST.format(
                            color = _config.bcolors.WARNING),
                        category = RuntimeWarning
                    )

            # reorder vars
            if(query_vars != ['*']):
                raw_df = raw_df[query_vars]
            raw_df = raw_df.unique()
            sorting_dims = []
            if(DEFAULT_ID is not None):
                sorting_dims.append(DEFAULT_ID)
            if(DEFAULT_DATE is not None):
                sorting_dims.append(DEFAULT_DATE)
            raw_df = raw_df.sort(by = sorting_dims)            
            return(raw_df)       
        else:
            raise ValueError('uhhh that shouldn\'t have happened')

    
    def _sql_query(self, 
                   query_components: dict[str:datetime.datetime|str|float|int|list[str|float|int]|bool]
                ) -> str:
        """Constructs an SQL query string based on provided query components.

        This function validates the presence of required query components, extracts
        relevant information, and builds an SQL query string for retrieving data
        from a database table.

        Args:
            query_components: A dictionary containing the following keys:
                - 'table_info': An object providing information about the table to query,
                                including its name and default date column.
                - 'start_date': A datetime.datetime object representing the start date
                                for the query.
                - 'end_date': A datetime.datetime object representing the end date for
                                the query.
                - 'vars': A list of variables to select in the query.
                - (Optional) Other keys representing additional columns to subset the
                           data by, with their corresponding values being lists of
                           allowed values for those columns.

        Returns:
            A string containing the constructed SQL query.

        Raises:
            _config.Exceptions.QueryingError: If any required query components are missing.
        """

        REQUIRED_QUERY_COMPONENTS = _config.QueryComponents.COMPONENTS
        
        # check for required components
        missing_required = _util_funcs.list_diff(REQUIRED_QUERY_COMPONENTS, list(query_components.keys()))
        if(len(missing_required) > 0):
            raise ValueError(
                _config.Messages.MISSING_REQUIRED_SQL_COMPONENTS.format(
                    color = _config.bcolors.FAIL,
                    obj = missing_required
                )
            )
        
        table_info, start_date, end_date, vars = list(map(query_components.get, REQUIRED_QUERY_COMPONENTS))

        # get specific table information
        table = table_info.TABLE
        date_var = table_info.DEFAULT_DATE

        # convert date time object to strings for the SQL query
        start_date_str = '\'' + start_date.strftime('%Y-%m-%d') + '\''
        end_date_str  = '\'' + end_date.strftime('%Y-%m-%d') + '\''

        # create argument string
        var_str = self._list_to_sql_str(vars, table)
        sql_str = f"SELECT {var_str} FROM {table}"

        # collect all filter clauses
        filters: list[str] = []
        if date_var is not None:
            filters.append(f"{date_var} BETWEEN {start_date_str} AND {end_date_str}")

        additional_sub = _util_funcs.list_diff(
            list1=list(query_components.keys()),
            list2=REQUIRED_QUERY_COMPONENTS
        )
        for col in additional_sub:
            if col in _config.KeywordArguments.QUERY_VARS:
                continue
            in_list = self._list_to_sql_str(query_components[col])
            filters.append(f"{col} IN ({in_list})")

        # append WHERE/AND only if there are filters
        if filters:
            sql_str += " WHERE " + " AND ".join(filters)
        
        # limit the number of rows returned
        if(query_components['row_limit'] is not None):
            sql_str += f' LIMIT {int(query_components["row_limit"])}'

        return(sql_str)
     
    def _db_params_to_dict(self, db_params):
        """Converts a database contents object to a hierarchical 
            dictionary representation.

        This function processes a database contents object, extracting 
        its class structure and attributes into a nested dictionary format. 
        It performs several validation checks to ensure the integrity of 
        the database contents.

        Args:
            db_params: The database contents object to be converted.

        Returns:
            A dictionary with the following structure:
              - 'Tables': A dictionary representing the 'Tables' class, 
                          containing its attributes and any nested subclasses 
                          (which are also represented as dictionaries).
              - Other class names (excluding 'Tables'): Dictionaries representing 
                                                        other classes defined within 
                                                        the database contents object, 
                                                        following the same
                                                        structure as 'Tables'.

        Raises:
            _config.Exceptions.DatabaseError: If the mandatory 'Tables' class is missing, if the
                                               'Tables' class lacks required attributes, or if any
                                               other classes lack required attributes.
        """
        name_dic = dict([(name, cls) for name, cls in db_params.__dict__.items() if isinstance(cls, type)])

        # mandatory tables class
        if('Tables' not in name_dic.keys()):
            raise ValueError(
                _config.Messages.NO_TABLES_CLASS_IN_DBC.format(
                    color = _config.bcolors.FAIL
                )
            )
        
        # validate that tables has the correct attributes
        table_attr = self._list_class_attr(DBC.Tables)
        if(not all(attr in table_attr for attr in _config.RequiredAttributes.TABLES)):
            missing_required = _util_funcs.list_diff(
                _config.RequiredAttributes.TABLES, 
                table_attr
            )
            raise ValueError(
                _config.Messages.REQUIRED_ATTRIBUTES_MISSING.format(
                    color = _config.bcolors.FAIL,
                    tab = 'Tables',
                    attr = missing_required
                )
            )
        
        # iterate through classes not Tables
        del name_dic['Tables']
        DBC_dict = {}
        for (name, cls) in name_dic.items():
            DBC_dict[name] = self._build_dict(name, cls)
        return(DBC_dict)

    def _build_dict(self, curr_name, curr_cls):
        """Builds a hierarchical dictionary representing a class structure and its attributes.

        This function recursively traverses a class hierarchy, constructing a nested
        dictionary that captures the relationships between classes and their attributes.
        It also performs validation to ensure that certain required attributes are present.
        
        Args:
            curr_name: The name of the current class being processed.
            curr_cls: The class object being processed.
        
        Returns:
            A dictionary with the following structure:
              - 'subclasses': A dictionary containing entries for any subclasses,
                                where the keys are subclass names and the values are
                                nested dictionaries representing those subclasses.
              - 'attr': A list of non-private attribute names defined on the current class.
        
        Raises:
            _config.Exceptions.DatabaseError: If any required attributes are missing
                                               from a class that doesn't have subclasses.
        """
        dic = dict()
        dic['subclasses'] = dict()
        dic['attr'] = self._list_class_attr(curr_cls)
        dic['class'] = curr_cls
        subclasses = dict([(name, cls) for name, cls in curr_cls.__dict__.items() if isinstance(cls, type)])
        if(subclasses):
            # subclasses present
            for (cls_name, cls_inst) in subclasses.items():
                dic['subclasses'][cls_name] = self._build_dict(cls_name, cls_inst)
        else:
            # no subclasses, check for required attributes
            if(not all(attr in dic['attr'] for attr in _config.RequiredAttributes.GENERIC_TABLE)):
                missing_required = _util_funcs.list_diff(
                    _config.RequiredAttributes.GENERIC_TABLE, 
                    dic['attr']
                )
                raise ValueError(
                    _config.Messages.REQUIRED_ATTRIBUTES_MISSING.format(
                        color = _config.bcolors.FAIL,
                        tab = curr_name,
                        attr = missing_required
                    )
                )
        return(dic)

    def _list_class_attr(self: LocalDatabase, cls) -> list[str]:
        """
        Returns a list of non-private class attributes.

        This function retrieves a list of attribute names defined directly on the
        class, excluding those that start with an underscore (_), which are typically
        considered private in Python conventions.
        
        Args:
            cls: The class object to inspect.
        
        Returns:
            A list of strings, where each string is the name of a non-private attribute
            defined on the class.
        
        Example:
            class MyClass:
                public_attr = 1
                _private_attr = 2
        
            attrs = _list_class_attr(MyClass)
            print(attrs)  # Output: ['public_attr']
        """
        keys = list(cls.__dict__.keys())
        res = [key for key in keys if not key.startswith('_')]
        return(res)
    
    def _extract_pragma_table_info(self: LocalDatabase, 
                                   cur: sqlite3.Cursor, 
                                   table: str
                                   ) -> list[str]:
        """
        Extracts and returns a list of column names for a given SQLite table.

        Args:
            - cur (sqlite3.Cursor): SQLite cursor object connected to a database.
            - table (str): The name of the SQLite table.

        Returns:
            - list[str]: A list containing the column names of the specified table.
            
        Example:
            >>> conn = sqlite3.connect('example.db')
            >>> cursor = conn.cursor()
            >>> columns = _extract_column_names(cursor, 'sample_table')
            >>> print(columns)
            ['column1', 'column2', 'column3']

        Note: Ensure that the SQLite table exists and the cursor is connected to 
                the appropriate database.
        """
        # Use a parameterized query to prevent SQL injection
        out: sqlite3.Cursor = cur.execute(f"""PRAGMA table_info({table})""")
        res: list[str] = out.fetchall()
        names: list[str] = [i[1] for i in res]
        dtypes: list[str] = [i[2] for i in res]

        # map to best fit pandas type
        for i, dtype in enumerate(dtypes):
            try:
                dtypes[i] = _config.TypeMapping.SQL_TO_PANDAS_TYPE_MAP[dtype]
            except Exception as e:
                pass
    
        return {'names': names, 'dtypes': dtypes}
    
    def _extract_column_names(self: LocalDatabase, 
                              cur: sqlite3.Cursor, 
                              table: str
                              ) -> list[str]:
        tmp: list[str] = self._extract_pragma_table_info(
            cur = cur, 
            table = table
            )
        return tmp['names']
    
    def _extract_column_dtypes(self, cur: sqlite3.Cursor, table: str) -> list[str]:
        tmp: list[str] = self._extract_pragma_table_info(
            cur = cur, 
            table = table
            )
        return tmp['dtypes']
    
    def _format_vars_dtype_DBC(self: LocalDatabase, 
                               strings_list: list[str],
                               max_length: int,
                               initial_pad: str | None = None):
        """
        Format strings from a list so that each line doesn't exceed a character limit.

        Parameters:
        - strings_list (list of str): The list of strings to format.
        - max_length (int): The maximum character length per line.
        - initial_pad (Optional[str | None]): A string to use as the initial pad.
            Default is the empty string.

        Returns:
        - str: The formatted string.

        Example:
        >>> strings_list = ['apple', 'banana', 'cherry', 'date', 'elderberry', 'fig']
        >>> formatted_string = format_strings_within_limit(strings_list, 20)
        >>> print(formatted_string)
        apple, banana, cherry
        date, elderberry,
        fig
        """
        
        if initial_pad is None:
            initial_pad = ''
        
        res: str = ''
        line_length: int = len(initial_pad)

        for _, string in enumerate(strings_list):
            # Check if the current string can fit in the current line
            if line_length + len(string) + 3 <= max_length:
                # If it's the first string in the line, don't add a space before it
                if line_length == len(initial_pad):
                    res += initial_pad + string
                else:
                    res += ', ' + string
                line_length += len(string) + 3  # Update line length (+1 for space or comma)
            else:
                # Move to the next line and add the string there
                res += ',\n' + initial_pad + string
                line_length = len(initial_pad) + len(string) + 2# Reset line length

        return res


    def _format_columns_repr(self: LocalDatabase, 
                             strings_list: list[str],
                             num_columns: int
                             ) -> str:
        # Thanks Chat :)
        # Calculate the maximum length of each column
        max_length: int = max(map(len, strings_list))

        # Calculate the number of rows needed
        # Equivalent to math.ceil(len(strings_list) / num_columns)
        num_rows: int = -(-len(strings_list) // num_columns)

        # Initialize the formatted string
        formatted_string: str = ""

        # Iterate through each row
        for row in range(num_rows):
            # Iterate through each column
            for col in range(num_columns):
                # Calculate the index in the original list
                index: int = row + col * num_rows

                # Check if the index is within the range of the list
                if index < len(strings_list):
                    # Add the string with padding to the maximum length
                    formatted_string += strings_list[index].ljust(max_length + 2)

            # Add a newline character at the end of each row
            formatted_string += "\n"

        return formatted_string
    
    def _write_vars_dtype_to_DBC(df: pandas.DataFrame, 
                                 n: int
                                 ) -> str:

        # Capture the output of pandas.info(verbose=True)
        buf = io.StringIO()
        df.info(verbose = True, buf = buf)
        info_output = buf.getvalue()

        lines = info_output.split('\n')
        dtypes = {}
        for line in lines:
            line = line.split(' ')
            if(len(line) > 1):
                if(line[0] == '' and line[1].isnumeric()):
                    line = [l for l in line if l != '']
                    dtypes[line[1]] = line[-1]

        res = []
        it = iter(dtypes.items())
        for i in range(0, len(dtypes), n):
            res.append(dict(itertools.islice(it, n)))
        res = [str(i) for i in res]

        concat_line: str = ''
        for line in res:
            concat_line += line
        concat_line = concat_line.split('}{')
        for i, line in enumerate(concat_line):
            if(i == 0):
                concat_line[i] = f'VARS_DATA_TYPE = {line},'
            else:
                concat_line[i] = f'\t\t\t\t\t\t  {line},'
            if(i == len(concat_line) - 1):
                concat_line[i] = concat_line[i][:-1]
            print(concat_line[i])

    def _get_yn_response(self: LocalDatabase, 
                         input_message: str
                         ) -> str:
        """
        Prompt the user with a message and return 'y' or 'n' based on their response.

        Parameters:
        input_message (str): The message to display to the user.

        Returns:
        str: Either 'y' if the user responds affirmatively (yes or y), 'n' if the user responds negatively (no or n).

        Example:
        >>> response = _get_yn_response("Do you want to continue? (y/n): ")
        Do you want to continue? (y/n): y
        >>> print(response)
        y
        """
        while True:
            inpt: str = input(input_message)
            if inpt.strip().lower() == 'y':
                return 'y'
            elif inpt.strip().lower() == 'n':
                return 'n'
            else:
                continue



def raw_query(query: str, 
              con: str, 
              return_type: str = 'pandas'
            ) -> pandas.DataFrame | polars.DataFrame:
    
    raise NotImplementedError('This is bwoken :(')

    query = query.lower()

    # only select statements
    if(query[0:6] != 'select'):
        raise ValueError(f'query: this function only works for "select" statements')
    
    # find index of FROM
    from_idx = re.search(r'\b(from)\b', query).start()
    
    # parse query vars
    query_vars_str = query[7:from_idx].strip()

    connection = sqlite3.connect(con)
    cursor = connection.cursor()

    if('*' in query_vars_str):
        cols_cur = cursor.execute(f"""{query} LIMIT 1""")
        cols = list(map(lambda x: x[0], cols_cur.description))
        query_vars = [col.lower().strip() for col in cols]
    else:
        query_vars = [s.strip() for s in query_vars_str.split(',')]

    # query vars
    result = cursor.execute(query)
    data = result.fetchall()
    connection.close()

    # convert from list of tuples to list of lists
    data = [list(d) for d in data]

    if(return_type == 'pandas'):
        df = pandas.DataFrame(data, columns = query_vars)
    elif(return_type == 'polars'):
        df = polars.DataFrame(data, schema = query_vars)
    else:
        raise ValueError(f'return_type: expected string `pandas` or `polars`, got {return_type}')

    return(df)


    









            


