import json as _json
from inspect import Parameter, Signature
from pathlib import Path
from typing import Any

from fastmcp import FastMCP
from langchain_core.language_models.chat_models import BaseChatModel

from walt.tools.schema.views import ToolDefinitionSchema
from walt.tools.executor.service import tool


def get_mcp_server(
	llm_instance: BaseChatModel,
	page_extraction_llm: BaseChatModel | None = None,
	tool_dir: str = './tmp',
	name: str = 'toolService',
	description: str = 'Exposes tools as MCP tools.',
):
	mcp_app = FastMCP(name=name, description=description)

	_setup_tool_tools(mcp_app, llm_instance, page_extraction_llm, tool_dir)
	return mcp_app


def _setup_tool_tools(
	mcp_app: FastMCP, llm_instance: BaseChatModel, page_extraction_llm: BaseChatModel | None, tool_dir: str
):
	"""
	Scans a directory for tool.json files, loads them, and registers them as tools
	with the FastMCP instance by dynamically setting function signatures.
	"""
	tool_files = list(Path(tool_dir).glob('*.tool.json'))
	print(f"[FastMCP Service] Found tool files in '{tool_dir}': {len(tool_files)}")

	for wf_file_path in tool_files:
		try:
			print(f'[FastMCP Service] Loading tool from: {wf_file_path}')
			schema = ToolDefinitionSchema.load_from_json(str(wf_file_path))

			# Instantiate the tool
			tool = tool(
				tool_schema=schema, llm=llm_instance, page_extraction_llm=page_extraction_llm, browser=None, controller=None
			)

			params_for_signature = []
			annotations_for_runner = {}

			if hasattr(tool._input_model, 'model_fields'):
				for field_name, model_field in tool._input_model.model_fields.items():
					param_annotation = model_field.annotation if model_field.annotation is not None else Any

					param_default = Parameter.empty
					if not model_field.is_required():
						param_default = model_field.default if model_field.default is not None else None

					params_for_signature.append(
						Parameter(
							name=field_name,
							kind=Parameter.POSITIONAL_OR_KEYWORD,
							default=param_default,
							annotation=param_annotation,
						)
					)
					annotations_for_runner[field_name] = param_annotation

			dynamic_signature = Signature(params_for_signature)

			# Sanitize tool name for the function name
			safe_tool_name_for_func = ''.join(c if c.isalnum() else '_' for c in schema.name)
			dynamic_func_name = f'tool_runner_{safe_tool_name_for_func}_{schema.version.replace(".", "_")}'

			# Define the actual function that will be called by FastMCP
			# It uses a closure to capture the specific 'tool' instance
			def create_runner(wf_instance: tool):
				async def actual_tool_runner(**kwargs):
					# kwargs will be populated by FastMCP based on the dynamic_signature
					raw_result = await wf_instance.run(inputs=kwargs)
					try:
						return _json.dumps(raw_result, default=str)
					except Exception:
						return str(raw_result)

				return actual_tool_runner

			runner_func_impl = create_runner(tool)

			# Set the dunder attributes that FastMCP will inspect
			runner_func_impl.__name__ = dynamic_func_name
			runner_func_impl.__doc__ = schema.description
			runner_func_impl.__signature__ = dynamic_signature
			runner_func_impl.__annotations__ = annotations_for_runner

			# Tool name and description for FastMCP registration
			unique_tool_name = f'{schema.name.replace(" ", "_")}_{schema.version}'
			tool_description = schema.description

			tool_decorator = mcp_app.tool(name=unique_tool_name, description=tool_description)
			tool_decorator(runner_func_impl)

			param_names_for_log = list(dynamic_signature.parameters.keys())
			print(
				f"[FastMCP Service] Registered tool (via signature): '{unique_tool_name}' for '{schema.name}'. Params: {param_names_for_log}"
			)

		except Exception as e:
			print(f'[FastMCP Service] Failed to load or register tool from {wf_file_path}: {e}')
			import traceback

			traceback.print_exc()
