## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.11.1/highlight.min.js"
  integrity="sha384-RH2xi4eIQ/gjtbs9fUXM68sLSi99C7ZWBRX1vDrVv6GQXRibxXLbwO2NGZB74MbU"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.11.1/languages/go.min.js"
  integrity="sha384-HdearVH8cyfzwBIQOjL/6dSEmZxQ5rJRezN7spps8E7iu+R6utS8c2ab0AgBNFfH"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-RH2xi4eIQ/gjtbs9fUXM68sLSi99C7ZWBRX1vDrVv6GQXRibxXLbwO2NGZB74MbU highlight.min.js
sha384-Kg8F1zN2geUZfk9LughWCT5378u7xGK7BKD6dAATYKlLUZY1drmcsyHySB5VNbEQ highlight.js
sha384-oLdnQ1gMzlJ7B7Z/Z6BN+THM/9f4ttUFMNj67N4+UMLwJsyWeY6FrmRd/9UWYjdt es/highlight.min.js
sha384-y1Zb37JASB/nh0y5AwzfHsCTjmXRxFW9mWzd+u/fWY0ZPpEYEH3PuYfpfIHVNbi4 es/highlight.js
sha384-PJ1psNJ2XMc5He8b6/c5li+Z35sYk63xdRPFwmN0ARHXTgBmeii7cFdSRKs7/hEZ es/core.min.js
sha384-YDAmRlpggmvyle/gQXXc1Ro1mZf05UW90b4NwoIbJNWCmvoI1c7UJAJYqIhhhuuL es/core.js
sha384-jLwoy3boLGElFnKGf70oE0qXwrxIal1KEKum645ymEqUpJVFKAccrMhO0Mfn93LI languages/abnf.min.js
sha384-cJxhHyra4xzu+Muq3teYAZ8evxMzI8YcL4NxN0xf74sLrEngjVMzroESjgHkfhoa languages/actionscript.min.js
sha384-XZ2VUstQUrySJytA56pgjjlqxorW2iMtF12VxlQaF2EKQTZ5S4OFlzdb5ya8NCoz languages/1c.min.js
sha384-GBGN3YSK+JJ90mRQ0Zk4jR91MS+uGMUfICgUFxUAnEH1L8YcW7LuTwCV5Cbl1AB/ languages/ada.min.js
sha384-JgzxBlecrKq/11S/coPo3z0Jrci8ZJUe0mTeOYscEqlnTCAlKzsntomzNn259fMS languages/angelscript.min.js
sha384-NAVE+aGPjSXIrt5LxakpP2qnE9khMpJfJFR7b9TswZOT6fH+aWUuvau1dvad0eF/ languages/accesslog.min.js
sha384-NsG10PVz8g3YFLj9ryIHJrqiEjGR4nnGH/XQulTzQ1HyhmBTTjiNSjCdAt4L43LL languages/arcade.min.js
sha384-ioKFBz7IK1srBqD2/ZtEKYTUj0yv0rnUVBdJ8CdlCnwzifYahEJCYGz5572vamDd languages/apache.min.js
sha384-t+KZ4ysrf13P4W8ZKi2mJA23U88IHcBag5QO86vSoeSx1+jgFdiTytii2yZakq66 languages/applescript.min.js
sha384-t67w5HJDUbNsGJgMXu0fyABJE4s+YQn0hpn78kGPC1NfenqQ/JIw4g9jy0QMJZn4 languages/armasm.min.js
sha384-CMXVrh9g9wY0bYvbr18r26rwp495RtmyLa4uqYrnVBmJGG4IlJcfAzmVQunjabDM languages/asciidoc.min.js
sha384-RnYUsno1l7zHZCyh6yqHhZFOGiOp+r0Z8pEKkH3uhNylRRkF4fzSle64G5u+hNNe languages/aspectj.min.js
sha384-uKoJCgVgVJP6fShoFeJyzhGcIcuK0GR1MhUeOkp8ku0LWJVxmaqf1LKmGC3VrAO5 languages/autohotkey.min.js
sha384-ged+H4JNXmGkHMYWfYfax1sk57U7EASWstnVP4n/jFJDScjuqV562RgxriO7o3Un languages/avrasm.min.js
sha384-zSP3zAOzNB2Vc2QlVOZyxuOXegbwyCPxt2F+ElqJ8ZBnemgGzQ2S8s13Yo82OIdd languages/autoit.min.js
sha384-Btfrp4G+gSmAkx2oUvKfCWME0NWD3FQvtn+XCF14ANrDIH8IMswR2fHsSw71+04g languages/awk.min.js
sha384-BbT8tZtvkh8HPXIvL5RtzuljBwI3gR5KIdYxZyYSyI5C/+KNAGdzAiexvmxuroag languages/bash.min.js
sha384-eiOqb/ds7X3JkCFzpDHXdBBL35GrdT1oz3+qs3L2GqIH8P0/BRhJ4AqYHFTlGhcx languages/basic.min.js
sha384-7K5EqwYt3lX54EzwiiQ9p+5dhxeGZzf1S4WtkJYwOagxxxQ2icIwwwsQOt7TqqTe languages/axapta.min.js
sha384-94UkgHDSwJ0buTSCitB5A+1J78lWjZ8m21kNLahv+OtoGcRXzq3elTs7Jj3vybNt languages/brainfuck.min.js
sha384-/ZwoIxhhRAm/U5bNbOz9XQDSWX9gGoo9xZ/sHuQCn2I1JATvXaWmdTJ+dGyKKs28 languages/bnf.min.js
sha384-tMmX0hBMZeMrZhX6dUNxA94/DNJLl70ao6qu2N9+b/6Ep9Y2e1pBzVjxtLygIB+d languages/c.min.js
sha384-Ytzh/yy03vBWF6kQnbJh0cK1M/4QlNDbCeOG4TM9HW/Zouh+TEUfl1AGA+Cfx1jD languages/cal.min.js
sha384-aOG/Ys8xqIR+ZSq0zKt0MIQGMoAd/ArFaTk1jQQFzV5zFTS11VOtoRVg+6+RdnVq languages/capnproto.min.js
sha384-23wpjZzgf0LAvrV02G/L6voTPSR7L8tbLMjA+JRsUwDzYued9iXYKmIiCZBzhfbo languages/clean.min.js
sha384-/NmZuRKMLmGdYdg6Mv3nuVCEh6wV4sqKILHurG9LCLbP8oxmcckzVHwHovKAqisd languages/clojure-repl.min.js
sha384-Y7lK00Ww4yHzRK965LfVd7rIbSKZOwyj/QpoAwDYvUZRqMPJ5d4dEwn8y/31nGwn languages/ceylon.min.js
sha384-OlTMbXKfvutaqAFG+Ok+n1N9bAIitJdnvLVaT6yAlSXZZWW7/5ejCKFqqVydeBf+ languages/clojure.min.js
sha384-PInWnmju5g7EmUG5eiPVUat7Psw4dFOQZCNdSV5R/HgXYI+6efBY9fTv+Ye9+kwq languages/cmake.min.js
sha384-e4CGoHQVPTtOEPw4tgt/eVQYVYAlQ+Z5Q/CYtYPFwaNm7sfXG0QYvAlsStTIR52o languages/coq.min.js
sha384-o69l95Xnl1QN1rzjvtdZz5nFzLKQFkKdtcBx39fQKqnbLnspj/dG9pi/c56d+F7b languages/cos.min.js
sha384-6tcHV8CgHkKMa5S21y+g9KLl9NcNtLLZWQh5dEXJLE0XTvyjWqNr/AHwcWYgb2b3 languages/crystal.min.js
sha384-B711MHXDqRvH/pKkxJk84RyRt9g0qyAJFsu2XukZKoCdnEiBmA6Aq9fO23ZCS7qk languages/cpp.min.js
sha384-8+xmzcwU/4KR7UGCSWSlp93D7KBBCLH0M+vKkk+INtjSKVzLAj6BNizH0QJZD10Y languages/crmsh.min.js
sha384-JO8fEvV87kSAjUzHsFz7JmXfX/6GSH9kMwAxCml5Hp3eNvXfo3Ze/0FJ5Xgyd9n8 languages/csp.min.js
sha384-Z+o7SU/ldIEIdOIqpMV+9s2n8EE1rZTFSRv5Sd7rlaSoPTpyflmmZ/oRb6ycw/2s languages/csharp.min.js
sha384-SYiBosECz9pils/io9oL5fJPPfdyeu9u370WHaOhVFSweUU5GedTHsp9a4tVg5p2 languages/d.min.js
sha384-7uTDEXhVKbXA/HcXr3PjZh6Ig9OOAu/ZpWDXkdK3k9Ke+KOdipYda3eR+vfRQgq+ languages/dart.min.js
sha384-04MxX6iQ0WrwX6Df4GJWGCXwfr5hVS5CQ0r9CS7aunho7Fkj/AAWbEPU8a6G+4LA languages/diff.min.js
sha384-k96k3i1YXsDJVihIVjW3r7MAHcD3tDMKjwwUsaBQSaSYA6uYO8RX0blGGZ4HVTm+ languages/delphi.min.js
sha384-hly+Rz036+A3/domxShxHoja13X3lfx8nyG3V8aMeOe7Efwu6gUaSrDxq9BKwYk4 languages/dockerfile.min.js
sha384-RI0iO3hAdUA6nZqyVrS82GoMXR0HcbzOTDO+9FNwVZz3NLJOFcbW3bPnRhrbNgGI languages/dns.min.js
sha384-krxOUFgiyevtYl4SsL416jqrUPU3K9W6fDe/TRvWeXwLYKWKq/f+XHL2AHwGYswO languages/django.min.js
sha384-s10f61nScVyophx7eDq30HRz4VUaiwMkHFL+yAjwFUh22Dw6ArIevPx07+r0Dgoj languages/dos.min.js
sha384-dxkj2dPKN260opX1T033+gUCKz2AsHi07XvsW9Uzf1i8PU5SoivwJ5N4HKcZNZSe languages/dsconfig.min.js
sha384-/jmxjIpzwhFsVwlnLFrxaoygMa+xnsd2V3qpcJnNrBHtat6bCuMB3Dsg/EyNp9I4 languages/dts.min.js
sha384-h+bHwQb+0wBE5seu/NF0fQC12p4bXHOqEgC++5HuL49ZpDj4Rs+sfAVnQwjrcrsP languages/ebnf.min.js
sha384-S22aGcA35DQpIM4Qxca+hj4rLFEeqis7uaQCfQXyfsBGy6qNct2Moy73CGgfVyG6 languages/erb.min.js
sha384-VlMxnZXYI9pxkOSQo4JxmDew+kDGDvKByohNyVUJkTn+c7MZ9bOpWC9AMLoD8ClE languages/dust.min.js
sha384-RjYkl2jsX2gc5mfnlUASkLzJ9r8BBWIIKLHIM1ye6kbWbZOZOrnv7eXu7ncuCqIG languages/elm.min.js
sha384-ePHZZw3pHRpCfVsGrMrlvhSvBYgE5hkpm7H77FGD0J5EJZYM600ezxR4SEOyrYhN languages/erlang-repl.min.js
sha384-MQHNkCaYRX2k6cd21j3k/t+rRxhTX1gZsd1ZqCVnfN9XePT+6gN6C5m01hPN0+0n languages/elixir.min.js
sha384-/RiBzQ7U6WaPqVGsPRMOZJJtA+uYrw8sV1x8BC9o6SzEXaq10F8ktmo/2RO3j87M languages/erlang.min.js
sha384-durukQKqWm/OLUWwhGWoCboff5Ad4zinmT2cxAlafR2dzQADI1hROL113HVZQwWV languages/fix.min.js
sha384-IiEC8fiTaLaSXXaeFq/XxoUse88fj4V+FmikhgzuXbiTLzfrfrWym3Pde4s2tv/P languages/excel.min.js
sha384-js87axhH2Q+jkxUjpaX7DgtCMTO45eaDvzB5d4JvbuVwLp0w3b9We/TfVy2qJaK4 languages/fortran.min.js
sha384-958lb77iWrvN6IRcHnaM/qHwHaye1Ns4YMKq0ufR7N27UGhSQ1BkXxDxBQTu0PpV languages/flix.min.js
sha384-htm9RGDfsS48hoyuALwxfHF9tE0+kcsYq54ft4i/11yP/YEByAFQMsqrz8uQDMp1 languages/gams.min.js
sha384-My7cG/oMwBIO196/j7VyG5E0S5Zc6i8e1PEnrw+cerlHsu2XIveyhg1NcBSV62gZ languages/gcode.min.js
sha384-zwprBsudDpk7ZM/Cj6Ei+edk9dGBrPk08hMPFJP2fod8+BFM7fY2KLpNCVgolqKW languages/gherkin.min.js
sha384-aY6qGj9ATJg5NQP+PKzvlPo0R9EpuMwMG+/nnfKvRooFjGIHirlf4+MNkQ27cSYo languages/gauss.min.js
sha384-HdearVH8cyfzwBIQOjL/6dSEmZxQ5rJRezN7spps8E7iu+R6utS8c2ab0AgBNFfH languages/go.min.js
sha384-55RCrjtjoduGgxOrAm5xxwlpSaef1T4sAUbyj0QiAl246lH+ZlanDaelryeGI/yO languages/glsl.min.js
sha384-kIR9nX4pHHuKEk9toPCN64qNaR9xDo3j4eeGtLp8n4cgLGReXFHi2eRSrooelgmo languages/golo.min.js
sha384-U8+Z32JqO6yTfhn2BL/TN4GeoO3cFoOYjMig0J5mRgC9ZCQTIGOux3X80SPjqlQV languages/gradle.min.js
sha384-M04f/a+xFV20/v8ZQLe5lPeqUKrH0A0h6HUSWFRvq4RE4xlU1yaJIE5XqNSuR2Ke languages/graphql.min.js
sha384-Tm1Og6d1FV5GOFgkfZWi7DcSeMEA05xEeZfifgFC5IH6xxfoXwg5PaNNxrX4yinC languages/groovy.min.js
sha384-yUMeM5KYbEcXVx/tTJykaXJCy1O0yB4Z8b+94jqNP8XTsjA/1fD8IZrSiDWdzCIz languages/haml.min.js
sha384-Wf7ONv70fvKjNB9G4vW90NGSBuI0RyC6/QrA/xRV8KQZWlhDkDarh4RL5sJlO153 languages/gml.min.js
sha384-Z7NJwTxw7Rkjd5VqLt3KR5VyFUlCThJa7/4khFskUKDYPBiuk/UcS10ApTNW2Xxf languages/haskell.min.js
sha384-iYbf8lXlprMiMUixdrRVRTObM8b+idJ8RuOJQkOddWFt9Ntk9xzMFSswSGaDHMNu languages/handlebars.min.js
sha384-y3QsspMBQ/tdNmj/WFWCMi03HPpxi/3zqEjBd4nvYNlmK6D5plkclAX02QobcoXC languages/haxe.min.js
sha384-xFcLkiDoLSMaPlMFSqWzTvG4CgeC7lb1Rn9NoaVMFdKVqhZqlHTlyOcg808xt2hs languages/hsp.min.js
sha384-lcTN4smcWkRAthjFCTBe/8gtM255uleMAezQLwzjHTWYk0mOt8mkhbqHb1CT++R6 languages/hy.min.js
sha384-X50fiL5mByDvJRwn0hkUXIEttF5t8hlEFSPUMq42KoryxgI4niflBsviuhahhWJf languages/http.min.js
sha384-NrmnsdarwteQHPGjt70kaQTi1KE0XfOJNX9/VJSg6wWwU6U2nPzjl3iWkgU1cvyx languages/ini.min.js
sha384-EhWKP2qG8dOnazobma+S+QoovF2lHsXdgJZzU+DCzLq3QXubzH6mZgGEVhxSTLcx languages/inform7.min.js
sha384-GmUqE3cyqLomj2R2YzFvx+nMHnHwaR2+CX9A0JjdxCxUma52cdJHMNa8KjL0dUfL languages/irpf90.min.js
sha384-9TY5qMANpFAepJtXQhSDxLfCBvsomeAr+sp0iYKmvF2J9rIEjJquOM07ktbHEt4t languages/jboss-cli.min.js
sha384-3C+cPClJZgjKFYAb0bh35D7im2jasLzgk9eRix3t1c5pk1+x6b+bHghWcdrKwIo3 languages/json.min.js
sha384-ahto23Rd35WcBQBqxp4l2zN3chmWVWXgz3C3q7NLclDLXuZ4UIQzF8m8wamO/U8k languages/julia-repl.min.js
sha384-IeWqKIkQi9YXMXP7u5Ty9njlj3ld2Yfhowbskgjb1D531UVvonFcNJAexMUjRt0c languages/isbl.min.js
sha384-cURSDNSYOEpA3JyB+Soin0iDfPlxgcO9kjA4sn9L3p9xMvFNFy414G6cx8MtkQ6K languages/julia.min.js
sha384-A8E38+eenPg19mMk5Ikv1fetaPL/yA2OsGR3hFs9TeeBX+zQ8u5bZNsPKmlFHwN4 languages/latex.min.js
sha384-xV5D+zLI1pZHwNZp99FxT+F/++F/2EOzF7nvoUK0oRxCQa/aoZVZaWSi4Qs7MkYA languages/lasso.min.js
sha384-8p3ylSU2l5lZhLCbVGbilhcrRrfNkkbPco4FHBplQT73qURGJFOJ5mdbuKQYvihp languages/leaf.min.js
sha384-/M2jjMTP/heMVduWFNDFah+SGfY+JtfWi94Cd7yZF0KmBBlX3VMni4S2HZmy/bvu languages/ldif.min.js
sha384-pY7nGT/bw5m3naX5e0wpghG20yHUEvSYY4r2+Azyu5e9hVhyv8UC2ZudZ6F1xKU7 languages/livecodeserver.min.js
sha384-XHf/F3AFtpwh/qvji8cbUFV7wLfrPKcyItaFlzg2/tS8EEBRgjCzWagzMOF4A6TP languages/lisp.min.js
sha384-H3eqxK8Ve23Rhvh2PBwboAibXmhLmgY2eTAg/ECOtrg8Z3vgwwHzLkhrc8m+jQtn languages/llvm.min.js
sha384-dbTI+BVfiAlIfjWMYrH83f/x/GYSKbujaX4g4F7q5YxbGtlS7qTLcwBQQvDdsGf6 languages/lua.min.js
sha384-DjL90zP08vzabGXs0CglFocqoxPXnzAcKhobGV+CQDA5FHGW7xCnxjylOhna+HB+ languages/makefile.min.js
sha384-IBR8syIivHAjWClvhCPJK/yYrjguspdPnkhz1c9WYOqbRpIqyU+uSqgBSdcL6e2U languages/lsl.min.js
sha384-lq1I9kuTT+eCtMpqcUeHyTatwv3ccB90NXm1b3I7ju0ueJKdD4rdRNMSGvhJpKBg languages/matlab.min.js
sha384-Rv26WbhHH4MDPzeExq4ECmZUYF942tlfVhqA91Drw1P+Ey55KjihLF9RJENxjWr1 languages/markdown.min.js
sha384-byJiBCS80ekFxsA8hSMUz/85cSGpR8iJLXtgA4ne1HoUhi9OhpAxiyUkvofK+EPk languages/maxima.min.js
sha384-aIunkp4Jt6FYn5B5JXJYmzv1JWzyH2EKndSk7FQTP2qTXTdkJ5Z9fn1sbmsZcxix languages/mel.min.js
sha384-30GpQz/MBTw8r3EMwZw9ZcO4mGil8Tpp42wILoZwk2p7nuue3MPUJqSZUFRN+clh languages/mercury.min.js
sha384-VeVYtm/0co5ibFqSd/kX+kGQTsiOZmyuQryNrHfS6CXUV5ZIc7xPvRH5j6S/L0+s languages/mipsasm.min.js
sha384-8SC7Ruwdvd1+SPlb2pRPLDB0z4yZ7f2NFCaxstWjVOe0juUxyfPcB8IW16y1asNM languages/mizar.min.js
sha384-9HA3U8M7HCd1WR5uvsHJj6bIL6KKRhs+/VXESj+fx19NzOWwOsreLPrfqcu0HEcq languages/monkey.min.js
sha384-Jp8o3RVLQbeL+zwk6HJj5u9ZtJqpOAYpEJoTWf9jTNAY7VbdWzpkn5Wqg/MT23j6 languages/mojolicious.min.js
sha384-o6wOqi/YrJTP1RwvLJToEivNExOLPJvFaJg9w75wM+oddFsOnXCZYBRd9SSrPpk1 languages/moonscript.min.js
sha384-MCymMQ0g245o8QQ0VtG8yqMky6BpRHO9440uy1W89uSYZ7tw9QtU0Ldkdsut2xpN languages/n1ql.min.js
sha384-mPlAhcndTZiSf/096MZYt0sKFL4wHbU7ZbS8Ef4ftXdZMW/imxwA1mUFFWsoupPO languages/nestedtext.min.js
sha384-CppArOy5YFkx1rJ0ZUWNnkF+CAV5OO9pHj3Fk7Kox4p/9H581UDmEck1buzehnjF languages/nginx.min.js
sha384-iPuHpT7TWALf59gGZeBuWFAHeSHgsQYvRSH+J5IZIQYFbUC/6t1L6Vco/++o4V5V languages/nim.min.js
sha384-bMeqX6rAheuWPwXKwIXApYaGX1pFpfFPXNi/LWdqIe5IkfyOY8xmtx0/QbwwyqO6 languages/node-repl.min.js
sha384-vTCx2kYVScZBBnkBfni8njIypRHzdROvkOGvNYuSMM9jRoKf4EjCgwEUSfbEyBuc languages/nix.min.js
sha384-azcLq84HapvEpXsDDJ2m1n7KovejGjCdGV4Ilw9xlcb6Yg2EyGNVr5dHZyoLdVDw languages/objectivec.min.js
sha384-Ox9CQ7lFdK85QgO6+LBGHukeEXutfvnC8BSMg78nFV2OqScEbL4inhxDrfcAFdmO languages/ocaml.min.js
sha384-uqzU/iZeZoBoO6lDL7csgDSfEo31LUjkuviLyLncwKMbmHABu+hLWTmfDFMv0Bps languages/oxygene.min.js
sha384-MCsZ5TWBR3TqVriMMD2GjKxQbPLpaD4p5AqGkEXJ3ilBX2C4dg5gs52S6DZGlPyb languages/openscad.min.js
sha384-1yS8ug1eUO+Z0OnJW7aQxVFgQdQ9a5P7z+Kg5QBG5mtm5LXykIWbR3iYpa7b5/7+ languages/parser3.min.js
sha384-PK5CVcMiWQ08dZFregTL56n1urRNEsSuWT6oiH1sFm/2ac/epI35hC5lx+YzjH5U languages/perl.min.js
sha384-KhwtWK055t7soEc/+7Xl/5OI6lzVGUsrQP4vRHzEMWXhHHpSLlgm4gGBMoMhkv11 languages/pf.min.js
sha384-MBJTfvMpjn9gjGPo9ywtyx6TJL2DqDdoAzzg1z2kWnf9UAsiE6AI2MBDBCL3zQ+X languages/pgsql.min.js
sha384-Z9EdtPaC8UiXHEq1WuQTdvqT+FwjLwaVTIwTCZW/dGfiU9nbF8Shvltrhqtw83Qb languages/plaintext.min.js
sha384-LIzUVMUAZRreWHTENKQ/wXuNK17VO4xPf+kR1a0aBKvM3S7vsedCcFJZC7N7vdDt languages/php-template.min.js
sha384-Bv/Sxv6HlOzYOdV1iQpJTG3xiqGWIIMq9xsFfEX8ss7oNWMgKqOa/J2WSFG2m7Jd languages/php.min.js
sha384-nBtPX/n+fZ4jiG7L334ZU7ctoqi/SlZgZSRqwr6LCAnkpEdViLfBlsX9RS/wmpUX languages/pony.min.js
sha384-PQTau0Tc43UlFKdcn1SMiWtnxDoEXsHx6KokysZ40OTg5yuKaCSGz4PYj4vZ1rlz languages/profile.min.js
sha384-hzilYICGbCQxAqe/4prYVmbugDPsbivG09sWqiOZ2ryCU2a+lkd4F+kJKSS5vRbT languages/protobuf.min.js
sha384-q/8iVbv95DiFN+l7qeoBwJ0Wju7gozJemMfG3DdxiOR8CfA/2dKvKA3W5t4l/n9t languages/powershell.min.js
sha384-Hw+JDJl3BaJE0DQ2qZUhbHrTi4UQ/je7PqMHPNfThOlI71MLMj8jRMr+wB0Fa9LS languages/properties.min.js
sha384-ODs5nLFYMjEkQw7tgNvYnGzUMfIEPHXpjYTbgdTE9SKYcqZayx7HFxx1xog022Md languages/processing.min.js
sha384-7efWBk7QdAk1TUr3ZC0YEw8aFov67m9G06Ojp0Yga1c2P+s05P7sxKkKIua7QyEO languages/prolog.min.js
sha384-66hg9EELNQEPzzK7tT42aA8jpsZEMQ7ZJAQS64lnGl+LcjCwlSyY8wKgLQWXcLjl languages/puppet.min.js
sha384-V9o4yDiTFuPL7DXmNGL/lWkau9FFJnPdVMWCD/An5TJ98wh8vJ8PpsM6AkFrY1cA languages/purebasic.min.js
sha384-n3iFvvEGhuJlnYpLj6JaCg7WiOhd8kQfKTZBDnpJwFZ0puRMhoq9JWtKDRw5Snyq languages/python-repl.min.js
sha384-HFXIcZ0LvZiSHVafazbK0tk3UqXJ52h+TDM7PXE9nwXdlGoQdEaO6jB8C0AwHVKT languages/q.min.js
sha384-KoaFZ+SurMDejYc+mOUksnuJhOvgUNwYVNu2g1/sfDxKPftLaYfUnryv6UPMxUOm languages/qml.min.js
sha384-poz1JEq/ihCmK+1p8IXPoluMlou+rnb+4Q3DenGB+mkNBB/JZkT0c/TERX8D+RUX languages/r.min.js
sha384-OLBjjst8j4l4BDrRVw4FPf6wGjnSsRWPQRT6TSbxtoxAwntVckZ19PFvq5Q9Fohj languages/roboconf.min.js
sha384-eLuKzru+2Y+h1soOz+wYwCZsrYosaTYclgIsHLtDwFszmM/8WYtOSsz8OSAWhlyw languages/rib.min.js
sha384-eXRt+aAa2ig1yFVDQCLis8k9s/1dikTcigj+/R07yNdIxc8BAG/b1uHDyEW3of17 languages/python.min.js
sha384-mHLK6S3KIR9o7V8VntnoT9SkqmPabl5olgxhGGKD42w8c7rkzwmMyfQ5nf2Lev+2 languages/reasonml.min.js
sha384-uDKJQOz02YyuyWWObvXPPoDWSD23uUFZvoSGyGnxx0YSnOO6VkFBnQb2SKXl4GeJ languages/routeros.min.js
sha384-l6Q6shA/PEP+ADFcUjhWcz36dJKPRbmt/mEQQ+wNV/8y5re620dXFnHM3IRukJWf languages/rsl.min.js
sha384-VZSbuLxWr3z9w+GVtpP2wE5kCfBDljCytWiFK6oxOsMG37y8fMM1nV2gkBiCFCkd languages/ruleslanguage.min.js
sha384-Rlnlnjp0sedK9HVa29DtCyVFVEDRZyeTMQ6+aOKUaXptJmpVGTEmCk6ziXfmku6l languages/ruby.min.js
sha384-kENps59cKQW5DV3vOEzpSp6tfGzWGpPYKz748i4gGziVSjieRtupNNu/WEwG3s8n languages/rust.min.js
sha384-iOnKe2LMBqjs68mlL/D1dkE5L+VIDjq12UUO1TyKepWMsj25PDJokYjsmaDrzqfy languages/sas.min.js
sha384-0wToQFz7iiWqirMz2oDLVp5zvMALcpCadc6YRGYOvT/eEum6RqzYfYFupG0wQQPV languages/scheme.min.js
sha384-1Gp1R5+JiIFAOpeRpk+ajfna9Bc8QyVBYdqwaso9mxfnseksw0wV6P+PKhU2TbGM languages/scala.min.js
sha384-5Ns/QgX27yZCK/QUcSUokhq05527gV3usf0YL6ePB2UeB8JsodWYX8GZe4GhxWfb languages/scilab.min.js
sha384-mSZF08WaP0Llc4GMwE0KA2V9yfZQimO5PvfcXf2AATDdqri3Q7IdV7pfbhVPJCHV languages/shell.min.js
sha384-2GrZ6a8N5i11PDfuj2MDvHVQppLDRwjSg1ns22QcwXe/xCliQwFwKItwovpu4w/M languages/smali.min.js
sha384-vTQfvc76nSatCi4u7So/dhhT+Yx+s4wEpy0iQeHWuBuCydVUrYhRw9ir10ep51p7 languages/sml.min.js
sha384-NX/WNuEVuD9lB5my4cHr1Y8mwOCGBNUOKoN5RirczkQHwStO59dPYpinzRw5awx5 languages/smalltalk.min.js
sha384-jrnLoVn13sB+/dTfoAYVPhg0tYGQzzuzSGP3WTk8OvKAY0hDejpUXFYYI3bohAyW languages/sql.min.js
sha384-aQKvTJSToV1aVHRtNEgpUcStt//3KHpzOaebJPh9WWSoIlLfXfH1mEWkmXG5tTrJ languages/stata.min.js
sha384-SHu8SsC+YDpzoAvLpey19YIUxPMOMXFR7ch5wrFLF3Ql0r1uXFUgRahft92sDxDb languages/sqf.min.js
sha384-+ErHpKxaFQvybqwEN9lNuFFYkOLFP4Vy3FZUM089R/HXXPO1Ngt2AI/YwrhwlNpI languages/stan.min.js
sha384-tGDA4o8Yz4itjdxfe9ZaPXLkHDmN24BeN6WYloLWCUcq/3ISFxz0srkA0pOkkS7z languages/step21.min.js
sha384-OXNa8wIt42aEgLXtReI/9YuiochL8Bde2XwTFECU3D28wAhZIy65FssjxPyqv+e0 languages/subunit.min.js
sha384-vvx4zgAYR25GypQyUKtqPPV9fFWSaddYg/4Ugg2xTMM3aj2C5GD6IrgDV2yD9RS2 languages/taggerscript.min.js
sha384-kboSn8zv4wAmWiuqYn/ezuIM3JlBZXrhwj6z9eAFjDRgIeOn7jI5jFD9//dmet5D languages/tap.min.js
sha384-ovH4H9AIdxlsR5A3ukEQZz8kLzWTzrR0kQiigLOheyc7ayp9B2RWfGQ1H7EahfEd languages/tcl.min.js
sha384-ASWlS+c35ZtDxfVp5FRLBLcP2K08bXUBNze9hBdaxSdK/TZNTuEVbNrm0JaXQjCZ languages/thrift.min.js
sha384-8x+WDkhIoa3jAPNEru6qCg2TIlnSnI1I0dflbbQkB0DrsgCguuCiH0c1E2bpsX/+ languages/tp.min.js
sha384-30no7nkUE3FF//bmsLgfaXy4ly5YfGERgKHQtWhJpS+Ppfm//mOKF3YiWcYcpf8d languages/twig.min.js
sha384-k9lKwR136qS4QirtT8X0VU9MsU3/fnrticZ2xGFiUzbMifXfQJp58b4CMzBRIMEv languages/vala.min.js
sha384-KFXIJ5qnoA95v2AANfmcsH5/wZRQWCc/08luJdFz69KdpQWCCSO0Cqe8EqM1NCpn languages/vbscript-html.min.js
sha384-9jaz2rSOFx5kYjZB+Loaf0a4ipf9Yvk08+8QskyozD+yaHdA14SgQKv0C52/UooX languages/vbnet.min.js
sha384-q4F1AGrMreQRFm4P6HrR+dLtRCcshPWWJ/0HW6ewizjoCBpBIsFxDVosvIHjLAY0 languages/vbscript.min.js
sha384-B6CpSqhvLdv8UUjRUUpu2xQ3FiWoV+c38uT2wWcj4IWj9wMyAK+AzQ0mKbfrVjwU languages/verilog.min.js
sha384-VlQzka731dwu27YSdIJ3tZrqCAmRLQx0FdETJM+V77rW7mxiFH+uR26/E0njLbkd languages/vim.min.js
sha384-J3pUKFGnHJH0czAle+lKF96F/08caYKJfTEzlt5dGbGTR9M4BwOeOqAgvSzsjOsP languages/wasm.min.js
sha384-UCgk+J6u0e/0TFYrhvHhyyoMAYuyJ6ATr+6CYWsxB0613LdtWr4dxNctmLDOmZyr languages/vhdl.min.js
sha384-d1w6as9peRTJh7Tgj50482oZIrj0+1guPVjy1QRfEafPvwMu6JZ/J9CiS5cT8XE9 languages/x86asm.min.js
sha384-kdC2tR9F+WuT3kdoIv5E/Xk5UbYbMN2PKaV6YCL73jTaEUu9XXQTE0OPzgFtD0GH languages/wren.min.js
sha384-Uiow6mnAgYX3te2byMLHtLeYI1Grz0vTcWl4ruu18ytHkJYJ4oNTxXBJooTtGt6M languages/xl.min.js
sha384-FQjSArDMJE4WMAJGcCNAV+IXIOljcIxM3UFAD2vxjedWmBnnDaAyqRG7AQHf/uM/ languages/xml.min.js
sha384-A/iMReLA0Bo3tLydBIoOQXQzYnrwL90jkHYUubrtERUGCbIuU7U0EHge0Xd2s5sr languages/yaml.min.js
sha384-wAt95BmWcuvBaWjMs87dY5wVTi/0N8M5dbK7Hho6MJroy2Pf7IklbS9PIgJwoJH9 languages/xquery.min.js
sha384-ap/DivPODkFgKCmv8VYYP5DoGWCuK0Rgw81vlQWffne5IStSzZJtRpn2SbceuxpB languages/zephir.min.js
sha384-fExacwevjaxWOjfZ4u5wK9AhGBB7cIpf9poF5tuE2ks+Cq7jrxdO6xMIWHtkExtu es/languages/abnf.min.js
sha384-DPMC6DNrQDnI19iz4boLDfCil/2VUOIvP95RdukjHU5GjXsdyGFymEoxIaskaAnq es/languages/actionscript.min.js
sha384-KkF76nCbG7BPU92q9wjUKzhE2og8d56vytlV+v+KZaYwfTgWQMAIeTfIqZLZyPvS es/languages/angelscript.min.js
sha384-nBHRn3msg3hmXvMrHK7V4nGHLgKG9i1/l/1/euhtIuXQLkkkpLPgCdcfRA3pru9Y es/languages/ada.min.js
sha384-UetTFEKURO3tHQVj0YC0xP9Lo84vGmqippXzOpf96Awpk5fTIJ1lrKDBZWj5LqXa es/languages/accesslog.min.js
sha384-7S47/0OrcUlxyT5U91QRrBl0WB3ngB4RbklzyUFqMvh5l6hQQ+aQZ5SNtmi2ssM+ es/languages/arcade.min.js
sha384-P+aXp77LDR3sReOm/LqJ/TZDajYl9YR0s5jPDLkAjkxbRCHBO8WhjPoWGXUYHoyz es/languages/1c.min.js
sha384-HPrpX+j31X1o9T5eF99pY7R2u5y8udMGQ5YFLMxlVqNmBsLu2vesOPKXGzEiNKcr es/languages/apache.min.js
sha384-bLfpGfjsTXF9t/A7zVpnKcKLVqfpoDQO3cj/Otmw7Kg6WeGCUKiKIfT4ORFq4YFK es/languages/armasm.min.js
sha384-xrWFlrFw2hKRyDUNdc/5RS4+OYmawKTVRBizj4keBc+hWQXYcwA7aI2T4pX5Jeoh es/languages/applescript.min.js
sha384-3p/AcS9vjLxqOK7hnymyAfIV1SJlGE0eW/iTNNfV56XtUc9kqpliCe3yJL7yZ2R3 es/languages/asciidoc.min.js
sha384-lNIMCZxm9l8gzr7WAxrq2w8h83FUIjC+sgZgfoMUOndXP96opihgLeKHWmwl1PcX es/languages/aspectj.min.js
sha384-KmgS3OVfpMEpVcANzQ7xAeLXR4gnymof90LNn09kuu78XoKmam7a4ElcX2Xl+u5M es/languages/autohotkey.min.js
sha384-ZC3UT/YWwRsO9wJC1OuvC1LbMr5Gl6m+8t4IWuxAVU7AdxXERm5q/mKZUc59L18u es/languages/avrasm.min.js
sha384-E9pJR2w9ceWfEDdQM0axW9Ko4/bJ6deNtqWCEhKxYTKLmMY/8kiQLvt5bT5koXKD es/languages/autoit.min.js
sha384-EHnfKoIoapCq56z5MHMlmX+azfja6Qd7pXAV3984kggFfPSWT9f9U96k/Yjm3dfy es/languages/awk.min.js
sha384-Z8A67hV3B66k1DT7o/OqP2m0DYQvowlpPFpnONn1omgUmBoqXPbxtd5yjwgn2APh es/languages/basic.min.js
sha384-Pg7b9hYE6kefjcNqAabhv8jOLCVoZubUaM4bZFjUJd0CaaQ14ksDI0GVllMtAF4S es/languages/bash.min.js
sha384-EfymdVdw4LpdoN7FCQD21aP38ryGq8spY2ym6nvau2xvQOiVXd2FZ1XgBWjVlWpd es/languages/axapta.min.js
sha384-5Di+hEgBSRkJwSrjHMQIs2QIwae05Eskbb7kizfL3EbgrGoeA9hxFwf52yTj5qHg es/languages/brainfuck.min.js
sha384-UqgBDF2X+N1GebyqNxeti8OeNEMW7n/TzYibf/T/xNJp/W7GZOlaFHLM/kxsSAuU es/languages/bnf.min.js
sha384-xaTVEdq02jgKStoYDcZD8NhTN1XV/TWpIu4OM53MtMiLl08+e9YJNENo+R/6Nwp0 es/languages/c.min.js
sha384-I2FUaDLnpeELCNAIxaoR3gkAhNdcc5Ame7AkbEMbwOsudd2VzF6qZtS7M2LQ1Ouy es/languages/cal.min.js
sha384-eS7SxRZ/E4cBZcxCUL0ytrdfO7Du0+I9h5sAbuO40XtFXPGEP3lrEZqygmHSuBdN es/languages/capnproto.min.js
sha384-1garsM3t5LF5Hgq5YEhQ69+QDOnHQhKwY6B/lR13gA/zmZw85MF0aTlfINZ9z/z/ es/languages/clean.min.js
sha384-Wknndeare8QK+wj3swbSm33Q/NPqFpAio/ALh1KoInphZ8nmRU6PFHl9cxr5E7UF es/languages/clojure-repl.min.js
sha384-2fm8xiTBnHGYikBmYzXsHdNkejBluscHr6lflc37uLEOmqS7mO5AWcEetfKofCq2 es/languages/ceylon.min.js
sha384-wUFe0FaJd5qqazVoR9bUsZ84IozJ/ADhmdglzzjnBMKPiQa4mdsh/Pu/Wbm06g/l es/languages/clojure.min.js
sha384-K5ORXxxMn6dMhashfWD47jLbEOmIAe4GJMVVBQUGhwQtXWz7ip1tAFpyWb8biVgO es/languages/coq.min.js
sha384-7HLwuzaDtJxWvaGQ7qGwWFgozJH1ljKCOPmIVeVosrCZ8RpzoA1wEF/6RSqDKyeP es/languages/cmake.min.js
sha384-WZQPy+Pp1cmMBgES65XN/920yzXprmouq01yN18D4DG2ImRPqtVA1Mtc0N09dhUU es/languages/cos.min.js
sha384-lans159An332J9CYcvdgkWOX6MHcoxH7Qu1lk25Kmc5cOTe39OuemUwkX8Ol2PfJ es/languages/crystal.min.js
sha384-+1Koxl0St78gEZW5CpFK+dbLp7yNsfwLzzQUsSGimV4k/RVJUz6YvqtsqtdbJyKf es/languages/cpp.min.js
sha384-7IiULxqEiwmhu0rNZ7NF+AYPbZjVxcSWYjLv41MAx+alnu8EfsHrJKMCMRaOHuSf es/languages/csp.min.js
sha384-Kre8OW+LeR9OVgElhVDWiEkl43gegrPE8SwOxGCBV2y0brdD/nsVN+vgnOa2zH+g es/languages/crmsh.min.js
sha384-xLfGW0hIBHie9xNFuVroNihI0BdEO8FKxOeCdyJBrO1eM7s5BsQ8F3fLtFydQZ+Z es/languages/csharp.min.js
sha384-1+sb+CiIZoPKZwgpbgG/uy6FfFwAIK9+SCzra3YzyxMO/WsCL2bweMxv2NnKEnUt es/languages/d.min.js
sha384-68y8u39VrtQ85zZP9c5coSo2jjPLBwTB1MAV4XLcP0220WLQivfhp6RVofQKWY+L es/languages/dart.min.js
sha384-joI34L4jMJOgkz6zOb3sqraHH5tmocRfXvs9HkdHfUpD3ceSxAqKlubpBT/4Q/sV es/languages/diff.min.js
sha384-vqP/jwtCqKD4HHL69aew2iS9gh2tMCjOJle+JqFmey4un+HuloMCh2GaaHVQy1eF es/languages/delphi.min.js
sha384-qyGdaK2usg+DEmSJUcr3Rogi6miy7A1Rn8QlRc71wBlMHpN1Y4b3d8hh2hf31dCu es/languages/dockerfile.min.js
sha384-awuLaeTnQUKMFxA81452r437c+WlKf1OGhsqRp3TG5i40wpQQe9w3ntPitWEfVcN es/languages/dns.min.js
sha384-c6VNDV3sM+ivvQX1L4LuDW3MjiYsC6u70/NKATWDBfdRODoECdSJ7oUhpcU/WBUi es/languages/django.min.js
sha384-GClxwhFSWyi58TL/Eu6ftCf/fsQtTyZrpLN+GhdaCqGnY46u9ECXsGFk0aRbD3hP es/languages/dos.min.js
sha384-crsqi4l7RdxidD6oicLW9AEb0v0F9FgySo2wRxI7rUi9uqhrox+vjtEsUsLmjnEI es/languages/dsconfig.min.js
sha384-K7yTU5VogT3E6tM+jSReX8uP5cDu3I6pWb9/aXFlCDObz0+wFjKaWjc5vUaBW6oU es/languages/dts.min.js
sha384-QVixKXA4tPisshbP4uNyLPdjm9n9p0/UX26/y6Qi5do5oJq2TiyTYLJuNXVqbUz/ es/languages/ebnf.min.js
sha384-QDkR9sKukFGwekCR3jkiiQCY4g9cQ0MAJLgPwPJoAqKz/VRtLef9Y6WxuGNRVh54 es/languages/erb.min.js
sha384-O4VN4oc9WRqTnhATTkN5YHdqGbSyzkB3KqOG/LsLKp2uQi9e8l10XT2pm3L4MNWy es/languages/dust.min.js
sha384-ZIzdDBJKRIOwooTaa0407JhlTryNn9kxMsoAeKytMGPbsZsDerzYigA+ht+A6C6a es/languages/erlang-repl.min.js
sha384-9PbcWGZ+dZWjn+bcdkYqto9TsTQ9+oT7QhJXDbVNDqxtOG2s2yNX+nSo8Zc88EwA es/languages/elm.min.js
sha384-/ISckXQDi5ORypxVU6Y3Zn9KkfUR+EzDTSsm72jsrhN2QHNZyfwtooynB0AC/gHz es/languages/elixir.min.js
sha384-4XvKs9FWdMZFUIHDBodcg+dK+cMy0apR3a2FnYO21omft3i1Ucx++DDGbZRnXCZO es/languages/erlang.min.js
sha384-zd6DV8PoDjE5VcuqShpnRa5BTGHpPW2QZW6Q9PwG1Zldb2tCKkeIIGAIztrs9Rkh es/languages/fix.min.js
sha384-PBfTDnXKYjfADHHbN2T7cmoiM0HI3MkM2+MczeGYiXBZE05RoSDZtGotmT/gm0gc es/languages/excel.min.js
sha384-YalUIgWEG8hsPt8z/fPqFxhZVrSnWpQO6mtVwkG5olnf6laeHt0fknWDfZuGiAUN es/languages/flix.min.js
sha384-A3+PvZy70E3+xm6wPPF7V67hbjr+poCeuK2CMSUJmhZcFvszluhlVveb2P7VCDBJ es/languages/fortran.min.js
sha384-uP0FJmKDMLeO97OswzEXwNfGic0W5BoctCIM6By7Qd4D6vxmB+2+WBDRwA82nMFf es/languages/gams.min.js
sha384-87y9y1xgycUkKd18av/nU/+HFKjbSYwl+bVSqKGBKm+DXXSkowVZMhI9MIHHLugR es/languages/gcode.min.js
sha384-3/4yfMv3zLjkiAH+ll0aNmDon3BL3/WHXQfsdVCZ0ypKndcTIvkIqnGBYDBO1Dzp es/languages/gherkin.min.js
sha384-i5vaYwafSNo5IdCPruLAzqJVisDe416NcgM6+nZn19Vl1mddKc67VO+IBPLl0z/Z es/languages/gauss.min.js
sha384-/UGh0AcfdC41Di2LsNVYCPOJ24RfaUWWXniaZoGuM52DaQR7fwStKAHJumI+u5yY es/languages/go.min.js
sha384-BAQNawDeoCUPCZWWabbjBwXCREsauI9uBgfmsygn0IxSgWsk3mnYN0rssUPupU4s es/languages/glsl.min.js
sha384-oyVD4OXJqCzAuLvR01l8RW90iC0xtJ5/qMUM8wVgDWv5Tz9AAimez46a93CdRXcE es/languages/gradle.min.js
sha384-lkFxRED+79+AKxnsztHuXVJjpvwjJALdq5nkp6ZX3bUgdATN9y6J+p+b+fSYSmkP es/languages/golo.min.js
sha384-ycqZEp/d+8Lmf+CQrUvmfgcQkd5TJPU36i1WFGNiK18rlap/VJSTcPfSj/ft5Faf es/languages/graphql.min.js
sha384-Xzi2eJDDOEBR7RxJQzJEnSDCYC2BexeUTC3ym8Vcvkv9Yie5Ci6lohBxIPRU8GJm es/languages/haml.min.js
sha384-eZ//tCVwtbperfgzjozR7tIfO8yR+hUoOZANaZnbxvLmMwpN/HzbQxAx3c8BuHks es/languages/groovy.min.js
sha384-sBE/3VGVCFFj0nSxyEPZmpYOSPN/4g8vLcp/k2RJ6qhYwNvywnQM3P926c2LpKGz es/languages/haskell.min.js
sha384-NLCQJcSNjba6pkZamXE5jwoHla3w/yqEOqVPG65LahN0IIVT77xBm6336aXLACrb es/languages/gml.min.js
sha384-Y3QRR1LoAMi3J7E26nRUoSLKhp1J85o9+C5V2bbNZzIxjxkpQSeaH9dIUfCfqWsJ es/languages/handlebars.min.js
sha384-GqfisVO1wjcEY5u2cu84aSOfDjBDkxJ+eUrNLLOlOv5/DE5BUMtoucazkLKc33Kp es/languages/haxe.min.js
sha384-xx1j8j6bchgaZE8CGBXi1RY4Ac5DHGissk8WNx5dFWvHaiR+JDT3So3Ec0O7FXcO es/languages/hsp.min.js
sha384-TOHxEWol49daRSqqVHC8wf1eHmSlRydKTVrPUDO7NvFTojs079H+wYoueSvlMQqS es/languages/hy.min.js
sha384-36ZwsK42N/jk3DquJeJr/r/oziBOtUxBcg0ZdTaaEDX+Zo/UMgBP4S2Sf4NEyq1y es/languages/http.min.js
sha384-CVynu7LzwkkAUiajSi0jprssYhgg9Vi1WSd8iR84Vmi6JdRGq3DT4vvEfjzoxxOK es/languages/ini.min.js
sha384-s7eptwG5YHTKDlhdwkEs8yADrpfilzo070AobA0MOprICKVooqeF8eBRbhLuIr4I es/languages/inform7.min.js
sha384-NMbxy0oxetMo/s0Hp2ueL1eMnwuhIDIVbPlNIaQ3Xu88Wnqgy17MCaHPF4cWz5BF es/languages/irpf90.min.js
sha384-AhwX6wPgJvIesTeqTwda/SmEe928MxabuP2v2HzBaZIlsu6usQWYUrlgyc30Arck es/languages/jboss-cli.min.js
sha384-UHzaYxI/rAo84TEK3WlG15gVfPk49XKax76Ccn9qPWYbUxePCEHxjGkV+xp9HcS/ es/languages/json.min.js
sha384-SYUm03vAGaCM27+4zps/g05YWHfdZdLurbeSG9cX8rvA91YG+46oCmvHq0sUdqZ1 es/languages/julia-repl.min.js
sha384-0J89NWtdtzjPKQxZ2inCaOGusyGoKh2PjP/THsEDiWT9m+4jEJh9RRN1ajClNDy1 es/languages/julia.min.js
sha384-Fa4dEuFJMVMiJLe34/RRfeWNlYAyUg003BDlRCiV4ghe9wtJt/gbqgLIylm5C3xI es/languages/isbl.min.js
sha384-bx4oC5pPCnrSzdPzDg3L9UsoJWiptFFwoGWKup6bRk82EzPFrquQa7T7I9iDwnXa es/languages/latex.min.js
sha384-4s2aahhd5CfrdDtAjQdSQk9y+ANaqNB1wJKGI3vrv5lV/UzPDq5d6RyRl11X6co8 es/languages/lasso.min.js
sha384-uHEfPpK2dquHTCzJ6+8r4dVRRYtgoPCuOpSSrNeHml8EZKzMo9jMZgqAJe/zh3Be es/languages/leaf.min.js
sha384-4TkjXWP4NhAgslZmqHc2Md4mEWtfJdqhZX6plhejzXyA5TfsbcPwqF9jhlPPzKWv es/languages/ldif.min.js
sha384-hHqqmua3IKhF6fr0H/xdaEufzkmqdtcmKuJXyV/xFUVBluA2gHk2A6+mM0RaOgRO es/languages/livecodeserver.min.js
sha384-kB1QfFMBwbQxnyNzZiWt6/rlLEODkTklgd4NfbEHc/8TYL6oV83o+Yydz8ZIx3Ie es/languages/lisp.min.js
sha384-5/rMDjrvVizUEgvQXzdMw4bEnPi4rc4g55j6uygbCX563PiwUWOt1DOFg8nWOW4k es/languages/llvm.min.js
sha384-HCBq0pjgKyOc3FNX31to33MxfNYza3HCbHLfWwdsnkH5r/VmmXTlRrvWSHTJyYvA es/languages/lua.min.js
sha384-iagI2W32nmVJq2EVd59zlgz2bFjR/MC0JRsbcFaeZL2gmvHzzOeyRTpTKGBSKd7q es/languages/makefile.min.js
sha384-1xa9vv3zbhKwAl+NlUd9gxmVMfg/tqSSkKLXgNrYpceYw6D3qboeyUbdR5FRiDh/ es/languages/lsl.min.js
sha384-1V+EHXAHVnBm9cEy3kpZDOO8LI9tOkHylYgT6gQdG5QRJVA8lIlLt/eQqfhiEPnr es/languages/matlab.min.js
sha384-E7UvgBH6skA1FIOcn3B2c68GtJzrmZlOOC5p/fsxwihTZG/bBedJZu5PC1+kGX7q es/languages/markdown.min.js
sha384-sS5Jp8d7Oj/yN+7MLDle3A1s1EMTAj5xEZY4JPEQEJHfeVJuYUbU6Cw36Bvtzbt2 es/languages/maxima.min.js
sha384-8HxIo9fQM7BlSbzN3UCuag0q/jYrHWniTSmrjQoSxRScXQE9irV0zqOijN0PSlQs es/languages/mel.min.js
sha384-F/Rc/u9CQvWQQDzZGuk5gcrErmpRpx8M/LEPQR5vWjlwwpvHAYcFz2o26jikU46u es/languages/mercury.min.js
sha384-KMMe6+QkGQP4DfvJsD56GpZJP/8yXSPru/9XyyyjWUcsMpLDC/244yr6u7v+xkRP es/languages/mipsasm.min.js
sha384-9AY/AWfeKuHfvubXxe0Ot+R/MUm6y4dIxtA9Y2VK4I62psmPT3SP7vbPjxELdgtg es/languages/mizar.min.js
sha384-5U/riLzJaveAAWNalVhH16Qlt1uKWO9OTfnszGIMB/r+3fNv738ALG9Bjt1b0g6v es/languages/monkey.min.js
sha384-SiTWKuB6c62R7pZmf1gcAZx7UmTcvAiPYKz27E9fApJ1IVJsEiCWbcTPrsvX44pJ es/languages/mojolicious.min.js
sha384-Xlyoa/4VWa+WAEFkJuwHtq28Xuq7bHs2f/Mnqa27PCLo23/gBCUbpUCmlSg4BxZo es/languages/moonscript.min.js
sha384-5iOnbrV6Vi7t9z/zZ3TOREhBYMR8voH0P4M3kBhRvaeJ1IS4GUkf6WJ1NHcaFy1N es/languages/n1ql.min.js
sha384-UHOC4FnNC/8bTChrsBexWVswrBnwU7nJluAaWhDvmaXphc+DKVcVjS6cQdOai4Tg es/languages/nestedtext.min.js
sha384-7QaPTK4CkHm5HP+HbJ7BwALXSAvCln1ofcgr0Myla2I3O6cU/pupqPnajKmyy03P es/languages/nginx.min.js
sha384-+wur+fHnc0LByCvO+YsWdEdTk+XsCzUK5xGmRsE9yKdr+tIVNP5FMCAjR3EVK94a es/languages/nim.min.js
sha384-DsfLkjXZZJdYYLjhxkRPvZcXjK9FKPE3Z8Rg9mX7sdt+H3lr8xQn/yp3ESn+3JD6 es/languages/node-repl.min.js
sha384-sHiwn92elWF7g0yS0TFr/A4SdK4j+sI7xZMoPlVn2DCR/6nBNyt+zCNVVcZvW4yB es/languages/nix.min.js
sha384-IA3s+KMOVf9zn8hH1+u4PCkGcunD2O3mNj4y10RDG/Wq7Gs7QWgiH6o01bjXeU1b es/languages/objectivec.min.js
sha384-K1wF0NzhmShaNd0mazLwrxnm0VT2wtNGclvFeDfuYRMM1g92YApMTm+Zknp7Jt+C es/languages/ocaml.min.js
sha384-RSz4nhHzUvfa7VZMlKCLhhmRHnCOI94hpLU22wroAbDo6ej8lyXvpoMaVy9Bu/BD es/languages/oxygene.min.js
sha384-i0I/GeD7Vmnr3ZHTheUZzjgT4G6zCKe29laq2JR3wMpAdg+OJEa/MPvToBYgesI/ es/languages/openscad.min.js
sha384-gEELuSKto/9yXfeeDtjld9wLeOXsB6dIqpR+3iWxs46sa1mrBTRYRtaFEYuJGoRo es/languages/parser3.min.js
sha384-doJqxKOhDf9Q9JYEBQXrTguSjzwPrbLiXJP2kdBkF+BYRNP55btwz9iH4b95mUTH es/languages/perl.min.js
sha384-bdNrhp/ANH5uQIZkLVgL+9WTzgGuL+KCTP0C1J9fjTyV/SHVt3MJWtHj8yT6yOT3 es/languages/pf.min.js
sha384-e9t/475eGSjNyO/O9vfEZHxdm21L6W2ZNlupK8+ejvTDnKDq96GjpdXw0z8/P++J es/languages/pgsql.min.js
sha384-OOrQLW97d+/1orj9gjftwbbQyV8LNAcgagqVKBhUYA08Hdi5w0p6VoB3yt2k7gnG es/languages/plaintext.min.js
sha384-/Btu11Y1N5hp8EDB6nZJCif5GWMrSC/bYMh0Re/SH64ZoH2qZRHUtqv1SFRPbNdM es/languages/php-template.min.js
sha384-VxmvZ2mUpp1EzFijS40RFvIc7vbv/d5PhMxVFG/3HMpVKD4sVvhdV9LThrJDiw9e es/languages/php.min.js
sha384-S1A1vOtq6uyFE7CdDMCgOh+QJUJNANVAE57nc0h83JuLf/KKnXV6DG5xYN9YeSGo es/languages/profile.min.js
sha384-7CvpdzhT3uZ5e8CYIfKANJE9bteQcSnZs6GFKag0gu2ff6PB5zzFvBLOucbevfH8 es/languages/pony.min.js
sha384-TLEjnB/5SQ3m7Vyn2l8PAHrw3XEapQeXk3i2z9nmdjsjoEj7TZ80bdAbhVRXS9Oa es/languages/protobuf.min.js
sha384-GZl3J+HUi40wO5QngTvTpHXB0M2gGDnn9E/MyHRRS4VEWDNCS96bODclYTX3/JId es/languages/powershell.min.js
sha384-/jY3EnpQIhHPBJqje2gBbmlolzDCUfluYNls4EgMz6K8V1eGkYfd1EJBM0tepwl0 es/languages/properties.min.js
sha384-FqTgfPEhiViMyKFbGpQqWEXPwIdKavGdohBylA1Wadm7R55wj+CK3ywF/i99S12F es/languages/processing.min.js
sha384-+4Gum6EgKQF+lIr6Kwgooj3H1iGUPni2Dt1Nw82217y7ukdLBABEKkRebOfh58iO es/languages/prolog.min.js
sha384-cnQI2r99QCtySILFW0UrIc7GsjgLgMqTA4DDvNXT8MIFv6R0FnIXgtvXZ4B9djD3 es/languages/puppet.min.js
sha384-JzVm9L5sHG1OkQUlukf6fAWc7lqxdgQxgrkHQAC3usU06D17VYQMbmG5pXeFHk39 es/languages/purebasic.min.js
sha384-3kYjpRu+I9WPa7JgdctnKgdzWP5dVkdMzFCUajkUvT8Tj/J5jzCgYywJKfxQixGP es/languages/q.min.js
sha384-CO+NiDedness48VlHW0FCIw9rQt8szFZ0IOXYiQo3LEAHLytXxM5NlKnmNYBUACQ es/languages/python-repl.min.js
sha384-q0gz/nlgzwxZWCJt0FETjTpy7edasOniV5Fpb199BRYO3fbF6lZwUrHp2QimGpJ7 es/languages/qml.min.js
sha384-WkipJgUBRzr12T7CtL8kF6QTuPHAl8f7Sx1YnmOm3KiVA/k3eDXfifuSvUW9M5nu es/languages/r.min.js
sha384-UisNo6B6CbdbM8eSkJZ/sOhC3INCQRd2Aa9PYPbJS9orEbLoXrz5/L2ZVdkBYjBY es/languages/roboconf.min.js
sha384-prjfJnxuqO34FF2Jbin9xg7fcUQ16dv1ZZsZtYjIPpRBCMR+NOYIZyBJ0TTkj6up es/languages/rib.min.js
sha384-ZV5sgX70bBgLkDR5Mtox5UsbJedBc39hRKPdvTw6miK4lSkE/wv94cLY2iyZb/sB es/languages/python.min.js
sha384-92LvADvD75pTTpftBNqDS5RSENaC8Vc2YkMwAZxQv7SIXBnBfyE5fGSe76jBOwDj es/languages/reasonml.min.js
sha384-lx4EyGnbY2/p5mdOwyYtZq1P3kDK+uYYXwIoOu/8CB4dXGAXRo0ey1d7lsK5CYd+ es/languages/routeros.min.js
sha384-LAhUTqmuZcIsUL/alfYk56B/EDJ6hxPJ4qw40aI2VtDi6y9h6FoOkjO0hkiKXwR7 es/languages/rsl.min.js
sha384-B9GZ4yX1vKeK+p3cECY4LysW1JwkSl+v9tcTuWrN6uFTl57TxNZAxtNIDz2ZaB+r es/languages/ruleslanguage.min.js
sha384-DpXpbYSP6sX4tcP61ZRjSMsnmF8V3c/hQILWjrGWI2g3lresYaqbxVxs+tioFMJn es/languages/ruby.min.js
sha384-JbkB8w/DGGyx29PIwSq8c/ZeiJB9T/X4mVAZFEyBiNlEAas98Q2NxpBPUlNIlE71 es/languages/rust.min.js
sha384-o1wXT68gljvlYb32XXeGTKoSMxzLH6idNrn14QDM6cCSLE+cSZ2rsHxWohlp94RV es/languages/sas.min.js
sha384-85l+nuYTwqCrz6ayLVJoKFDgWMnoLU5d9CkZueTjSVKsF4NVN/rTCMUX2eja75DE es/languages/scheme.min.js
sha384-1RdooMJZXQ2dDH5g5l2Hng32v4pYdGIseVv35/avPgsDAK1vHoXOS4usBVRfGx7M es/languages/scala.min.js
sha384-ucErcciHIbGrvsgkJbgQnWsVw4ECUkEL2iVMQVEvDSAWN03wdzSfKY+BZDmaV03E es/languages/scilab.min.js
sha384-u9PV7oWG/lZlm+GnftX7kn0w4b8rRfFxSv5SmJJPHWKGI03rz6VLqgZdQ1B5ez6b es/languages/shell.min.js
sha384-O0Inb820+EuZH82ASj7FVKYAmXRz+AzJ5zradHpgFEewAMxQ5xU85u2LjzhtQDho es/languages/smali.min.js
sha384-Swn4jjB/01DY9NpEJrtQxUzD+kUfzNnvZRHXcpgAiSpH4olKApMA+By4Z6Vj4AQS es/languages/sml.min.js
sha384-Z1UY7YWCZvtnfRfAmAfTHj9grARZMNCHeUsNQXEremgWKTp/xPLNWd9LjicW7N5Q es/languages/smalltalk.min.js
sha384-y25cn06synxhYnlKVprZdpakuFWVrm2jvn8pqiF4L85a05CI/6bNeT2+qXbUYIyW es/languages/sql.min.js
sha384-SveD0lu+sQ19Pj1EvRLK8JWhE1sOkBl1BpxuxdicTBhleJ/q3iuIIBDXeXlpUnGP es/languages/stata.min.js
sha384-c+PByV78Gx6rzMeQUAiFLkFI3yiikAvDXskx9wzUKzxtBj+wMMrXwF31PBpe5itr es/languages/stan.min.js
sha384-NxE+qqq3Ct3INUXh6gybGlkZxLzcnO4+v7jcNSLToh+jZwpOUTBF7MCvjq3RHtb0 es/languages/sqf.min.js
sha384-lujkAJIAmNFOmFbG6hXIpB8XOTRSEHP4r6EWuC5BASjcBOTGTlfY/hli/X5ED0sH es/languages/step21.min.js
sha384-cfbjywWjE/0O5rl9IJHuYMuTqsDXall2g2UaSpbCTJ72sHn3ECsmryYQIsv4Co4D es/languages/subunit.min.js
sha384-PqRy+7XBjDhelv+uK54dDkObcPZe2AcQaIqmmB55CNHD6wzvxDvBJquzWCyTrBLq es/languages/taggerscript.min.js
sha384-SPJMC5mp8MbNrBDcxGFtPEST2CrBmQsKK4R0+9VFJTXsU3JmGGhWIaK/mN76SwKe es/languages/tap.min.js
sha384-aI1neNyUcnd5KlqN8ryRQujCHdltCLh51YOKmgU4q/r6n3zLDy7lECkrCS2ZoXs9 es/languages/tcl.min.js
sha384-KrmABoGvDiD8CG93v/CkivllaA6WpR8Em2uBz1ljIo5+/kYYpA01zrgucp3sk+Hz es/languages/thrift.min.js
sha384-o/dElfOn171tHMMjnmogls5B4+u5D/YAYBABOLu+4xvwY+z7AwTzmZou1zy7rECm es/languages/tp.min.js
sha384-xx69PXqN0VBmdg5u/nxD+yHl24nb3B+/kAbMwwo5CF7mWQCHvlJ57zUYRRosjRSS es/languages/twig.min.js
sha384-OIeJ9bZoeT3bq6Fr0UsjyDwbFT/ur4Ng3gAvTRBPP+WlWroaR5y1DkRVvy0DrDp0 es/languages/vala.min.js
sha384-delh3ooaionw8Ky+Tix4I5BLmMYfezQEh7v4fVbmPCN6QbT8XUmkq3N0Wjg6hav2 es/languages/vbscript-html.min.js
sha384-ztGjA/YtvuFeKJwRqtrvV+3no91MVzQIzG5kQhHiIndfxmQOAum+vaaaDfA8Mg8U es/languages/vbnet.min.js
sha384-ZVOU05lYlww5C9caC+fbHEJd7urtWFvlMpItqceJD/yo3G0n5SKD8nYNGT9sGHYR es/languages/vbscript.min.js
sha384-GOec9CUPkagNxXeKKSs6bjAA6LyYrTj2fSfXFZTqp4rDvdhnr2Q8k9s+4P+y4W0I es/languages/verilog.min.js
sha384-RoYI/9AQK5AF2G++68SRqb7LiSal0kL5sm7eTfGI+cvx1PaJDbLr846Xz+Y1IZ4N es/languages/vim.min.js
sha384-YBbT3eXpwj5Ddx0MS774iycYICw4gZ1Rs1ExYGIdcYC4EJhaWsfd9uNlaJZBuDaa es/languages/wasm.min.js
sha384-brlrwbzJdrEDjIV9RTpTVBQfseypNb5xoAIsc9Y8uCXx9vomFKWiuPF/6pktK7H7 es/languages/vhdl.min.js
sha384-hM+bHIycWGStQJOeaWw4p4eza4RV/cA1SnM+RAHDfmgQb6omDjunt3JAX+UkSt0y es/languages/wren.min.js
sha384-6jCvxwz6+4SETxn2V7LPLbyOa80MAobeFcGMoSk3ZR25GKcQQdZierMKqC5XdM+6 es/languages/x86asm.min.js
sha384-46FNEC6s3e+Fb65sNaO1YzppTYYV5F1C9nkbl7RPU28RSeiCFzmUMII66uyUahqJ es/languages/xl.min.js
sha384-PQrsaWeWrBiE1CFRw8K335CaJuQRTjDGm73vn8bXvlwaw6RyqWObdvMTBS8B75NN es/languages/xml.min.js
sha384-4smueUtgWTorlNLbaQIawnVCcIAuw82NetPOGWN5PbZT/pMr0rjvZXj0EUzJV1nr es/languages/yaml.min.js
sha384-flK/DTPAyevTVkoaDWdCnHV4CGBqxtFP08E3gT15S4oXlRXMpce91+b05AoK3Tbc es/languages/xquery.min.js
sha384-PO1dx0p1z7LkGuamWi2nA1sl6a5XfEnTnjNuBDz/hpTLJ362Gi/8cNkvail0DiU7 es/languages/zephir.min.js
sha384-Fh/jSSfDM8TBxrEe/yPhSI9FpoMT/3TL5ib1pmehS1cHspQ9SWRYbUSW5QCCbI56 languages/arduino.min.js
sha384-KJxuHuiSsSbNteoBD1crG6mq4q1sF6HO7eOKEpp1aZ9LBc9W27jh4QxyS1gwj5rs languages/fsharp.min.js
sha384-y8fkIUXy48IaLjKcQF4+JHp/PLBAVmzN22hNtYyKvBI9H1V+a/Ddu55qNYdxeApL languages/nsis.min.js
sha384-WSCnMFB4pJP36Z4X6+v4yxBQBZmulTaHCPnFMCAnv89csu2oia55pCmdEtziYAR5 languages/coffeescript.min.js
sha384-tPOrIubtDHoQU7Rqw0o88ilthGO0/4xEZGB47XrQKWhrc1/SchwsDx+AP74u4nk0 languages/javascript.min.js
sha384-FJYXOzB9VRvClv0YnEy5AjVld4/Y2GM8Xq91Sqn/bwQyliXk03gMZCP/M1C0rJGo languages/livescript.min.js
sha384-0XGvxIU7Oq1DQMMBr1ORiozzBq3KpZPE/74mJysWRBAop1dZ9Ioq/qRWe8u30Ded languages/css.min.js
sha384-M7Wfa4KRyfGnn1i52Nqpr5zWcrmVaO0luxCB+2Txz1eI2FRQfpDcNimn1f86K2Cp languages/less.min.js
sha384-BYdYy4D3IX6eNNlKqsviUjxC6EqavvNwCVDMzmie3QXyArWdCQf+VvvFo4ciaNaW languages/scss.min.js
sha384-e+59xEZvRMXSRGD31B3HOBGAGqhhs+bbkxCqPuJDkSX5QGneIGTIfwdYJckTN3AO languages/java.min.js
sha384-7abn027YsNDPFilpW9aLlNUanPrq7Ht81zKQL9MKQq6/nkKrLczChRK5OA8GSKep languages/kotlin.min.js
sha384-+LCddHx4GBdzLLANkRhAIIxSWKewLIbk2nVReuACLcFRhYXxi/fhoaRWMYr1kOfo languages/stylus.min.js
sha384-5fqRyRDjJwFL5ud85A5a43LTiHQYvuRA8aoZQfQmjEKTQzQ58xBXhKVxno6hb/yS es/languages/arduino.min.js
sha384-cFMVFv2j0WFmDOILSH3KABqXK8eMV1cRhgjAcQXuWbMQZXvFPJxtftyUx89oi0sZ es/languages/fsharp.min.js
sha384-y1+UZ2Tj8YGxk8pltI2Xo0e17nA9sBMyynUZGHY1ZUDaVMBQCB91/ci3nYxZbJ4u languages/mathematica.min.js
sha384-xtCTpi4/d+FMxicqc3TLWv7AHZnIrRlzx1V/J9HGyTo+cOiu+DiV1xVbf+nPNNX0 es/languages/nsis.min.js
sha384-nC3NYYR2DoB1VGCbsComaX6JrgUGlYySu0O9lOl2YyM6bcZQYMvGDk8ecgQ5BUp2 es/languages/coffeescript.min.js
sha384-f7huPivS1dV2T5V+g0aJpgsY7WBHWCsioIq30tpNoXGizD65fWJYGuXXVPNI52VB es/languages/javascript.min.js
sha384-1D7DbOic0Z5nM2ldSO9O/EsBfsg/5x7X7So1qnMgscI2ucDevptcg7cTvrD9rL0D es/languages/css.min.js
sha384-vdVWblWyllqkopZzYcac6J7ynMUPiRdBDRGRezI2Ez7SE3xQUxjhPGs4lL2k4isR es/languages/livescript.min.js
sha384-EJ7n9HlCUKgtcBomJlrocJe2M2WegUc2r/TqymQdykuxcLeA25bQ5665qN58BWki es/languages/less.min.js
sha384-WMy5VYgOMFAnHhPJXVDCQ/Y/QPlUrBqNVPtFH6/gGg2F4uaAowyQ0y/9zWEeGpJe es/languages/scss.min.js
sha384-5GpB6kfA2w03pZhAUmmNSYvR5pLvne/Rzqc22BmHv+t9ES7ifMX/ZE7x5TBeqW4d es/languages/java.min.js
sha384-+aJFpyNBGTRiXRDN6BLrctauQBKExwSosxOiHLUYReXKTsckW/RgMavqX6W+zTBL es/languages/kotlin.min.js
sha384-uh4kSeG7zfeOKlocPsgg1BwvcZRX1O8AcBEmzJhxOSmKeHLdWFPbSmvU7hqGnPvP es/languages/stylus.min.js
sha384-s9OhOZLci2dao+WT5xmnxIgAID9YQRqb/zWb9UoLnO6p41wXb2/0HlhB3pgXFk+A es/languages/mathematica.min.js
sha384-CCauhmYx0fwWViYO6uiTII5shLTfiY/OzxKmLRTeCp8Ok81I2nXZS2Gb9lJVOSPC languages/swift.min.js
sha384-df1w1nJ43GNwmgbSCrT8YFIYyqFAm+lzj+b6ofuziX8Cfdg9QHFwbORDgAaj//wi languages/typescript.min.js
sha384-9NAaCxdhTO7TX6fYeUHyt+NC3ledirZOADyWdinDCTN1taeqj8sLLYqjE8YMf4Na es/languages/swift.min.js
sha384-Tv4mr9B7b+x2IynRXW/xcAxUj1+AoN9zyp0n9BWE1Nle3Zfm/zUeEztNLhIRjgE7 es/languages/typescript.min.js
```
