import subprocess
import json
import PIL

from typing import List, Dict, Any, Optional, Union


from jinja2 import Environment, FileSystemLoader, Undefined

from sqlalchemy import create_engine

from npcpy.npc_sysenv import (
    render_markdown,
    lookup_provider,
    request_user_input, 
    get_system_message
)
from npcpy.gen.response import get_litellm_response
from npcpy.gen.image_gen import generate_image, edit_image
from npcpy.gen.video_gen import generate_video_diffusers


def gen_image(
    prompt: str,
    model: str = None,
    provider: str = None,
    npc: Any = None,
    height: int = 1024,
    width: int = 1024,
    input_images: List[Union[str, bytes, PIL.Image.Image]] = None,
):
    """This function generates an image using the specified provider and model.
    Args:
        prompt (str): The prompt for generating the image.
    Keyword Args:
        model (str): The model to use for generating the image.
        provider (str): The provider to use for generating the image.
        filename (str): The filename to save the image to.
        npc (Any): The NPC object.
    Returns:
        str: The filename of the saved image.
    """
    if model is not None and provider is not None:
        pass
    elif model is not None and provider is None:
        provider = lookup_provider(model)
    elif npc is not None:
        if npc.provider is not None:
            provider = npc.provider
        if npc.model is not None:
            model = npc.model
        if npc.api_url is not None:
            api_url = npc.api_url

    image = generate_image(
        prompt=prompt,
        model=model,
        provider=provider,
        height=height,
        width=width, 
        attachments=input_images,
        
    )
    return image



def gen_video(
    prompt,
    model: str = None,
    provider: str = None,
    npc: Any = None,
    device: str = "cpu",
    output_path="",
    num_inference_steps=10,
    num_frames=25,
    height=256,
    width=256,
    messages: list = None,
):
    """
    Function Description:
        This function generates a video using the Stable Diffusion API.
    Args:
        prompt (str): The prompt for generating the video.
        model_id (str): The Hugging Face model ID to use for Stable Diffusion.
        device (str): The device to run the model on ('cpu' or 'cuda').
    Returns:
        PIL.Image: The generated image.
    """
    output_path = generate_video_diffusers(
        prompt,
        model,
        npc=npc,
        device=device,
        output_path=output_path,
        num_inference_steps=num_inference_steps,
        num_frames=num_frames,
        height=height,
        width=width,
    )
    if provider == "diffusers":
        return {"output": "output path at " + output_path, "messages": messages}





def get_llm_response(
    prompt: str,
    model: str=None,
    provider: str = None,
    images: List[str] = None,
    npc: Any = None,
    team: Any = None,
    messages: List[Dict[str, str]] = None,
    api_url: str = None,
    api_key: str = None,
    context=None,    
    stream: bool = False,
    attachments: List[str] = None,
    **kwargs,
):
    """This function generates a response using the specified provider and model.
    Args:
        prompt (str): The prompt for generating the response.
    Keyword Args:
        provider (str): The provider to use for generating the response.
        model (str): The model to use for generating the response.
        images (List[Dict[str, str]]): The list of images.
        npc (Any): The NPC object.
        messages (List[Dict[str, str]]): The list of messages.
        api_url (str): The URL of the API endpoint.
        attachments (List[str]): List of file paths to include as attachments
    Returns:
        Any: The response generated by the specified provider and model.
    """
    # Determine provider and model from NPC if needed
    if model is not None and provider is not None:
        pass
    elif provider is None and model is not None:
        provider = lookup_provider(model)
    elif npc is not None:
        if npc.provider is not None:
            provider = npc.provider
        if npc.model is not None:
            model = npc.model
        if npc.api_url is not None:
            api_url = npc.api_url
    else:
        provider = "ollama"
        if images is not None or attachments is not None:
            model = "llava:7b"
        else:
            model = "llama3.2"
            
    system_message = get_system_message(npc) if npc else "You are a helpful assistant."
    #print(system_message)

    if context is not None:
        context_str = f'User Provided Context: {context}'
    else:
        context_str = ''

    if messages is None or len(messages) == 0:
        messages = [{"role": "system", "content": system_message}]
        if prompt:
            messages.append({"role": "user", "content": prompt+context_str})
    elif prompt and messages[-1]["role"] == "user":

        if isinstance(messages[-1]["content"], str):
            messages[-1]["content"] += "\n" + prompt+context_str
    elif prompt:
        messages.append({"role": "user", "content": prompt + context_str})

    response = get_litellm_response(
        prompt + context_str,
        messages=messages,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        images=images,
        attachments=attachments,
        stream=stream,
        **kwargs,
    )
    return response




def execute_llm_command(
    command: str,
    model: Optional[str] = None,
    provider: Optional[str] = None,
    api_url: str = None,
    api_key: str = None,
    npc: Optional[Any] = None,
    messages: Optional[List[Dict[str, str]]] = None,
    stream=False,
    context=None,
) -> str:
    """This function executes an LLM command.
    Args:
        command (str): The command to execute.

    Keyword Args:
        model (Optional[str]): The model to use for executing the command.
        provider (Optional[str]): The provider to use for executing the command.
        npc (Optional[Any]): The NPC object.
        messages (Optional[List[Dict[str, str]]): The list of messages.
    Returns:
        str: The result of the LLM command.
    """
    if messages is None:
        messages = []
    max_attempts = 5
    attempt = 0
    subcommands = []

    # Create context from retrieved documents
    context = ""
    while attempt < max_attempts:
        prompt = f"""
        A user submitted this query: {command}.
        You need to generate a bash command that will accomplish the user's intent.
        Respond ONLY with the bash command that should be executed. 
        Do not include markdown formatting
        """
        response = get_llm_response(
            prompt,
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            messages=messages,
            npc=npc,
            context=context,
        )

        bash_command = response.get("response", {})
 
        print(f"LLM suggests the following bash command: {bash_command}")
        subcommands.append(bash_command)

        try:
            print(f"Running command: {bash_command}")
            result = subprocess.run(
                bash_command, shell=True, text=True, capture_output=True, check=True
            )
            print(f"Command executed with output: {result.stdout}")

            prompt = f"""
                Here was the output of the result for the {command} inquiry
                which ran this bash command {bash_command}:

                {result.stdout}

                Provide a simple response to the user that explains to them
                what you did and how it accomplishes what they asked for.
                """

            messages.append({"role": "user", "content": prompt})
            # print(messages, stream)
            response = get_llm_response(
                prompt,
                model=model,
                provider=provider,
                api_url=api_url,
                api_key=api_key,
                npc=npc,
                messages=messages,
                context=context,
                stream =stream
            )

            return response
        except subprocess.CalledProcessError as e:
            print(f"Command failed with error:")
            print(e.stderr)

            error_prompt = f"""
            The command '{bash_command}' failed with the following error:
            {e.stderr}
            Please suggest a fix or an alternative command.
            Respond with a JSON object containing the key "bash_command" with the suggested command.
            Do not include any additional markdown formatting.

            """

            fix_suggestion = get_llm_response(
                error_prompt,
                model=model,
                provider=provider,
                npc=npc,
                api_url=api_url,
                api_key=api_key,
                format="json",
                messages=messages,
                context=context,
            )

            fix_suggestion_response = fix_suggestion.get("response", {})

            try:
                if isinstance(fix_suggestion_response, str):
                    fix_suggestion_response = json.loads(fix_suggestion_response)

                if (
                    isinstance(fix_suggestion_response, dict)
                    and "bash_command" in fix_suggestion_response
                ):
                    print(
                        f"LLM suggests fix: {fix_suggestion_response['bash_command']}"
                    )
                    command = fix_suggestion_response["bash_command"]
                else:
                    raise ValueError(
                        "Invalid response format from LLM for fix suggestion"
                    )
            except (json.JSONDecodeError, ValueError) as e:
                print(f"Error parsing LLM fix suggestion: {e}")

        attempt += 1

    return {
        "messages": messages,
        "output": "Max attempts reached. Unable to execute the command successfully.",
    }
def check_llm_command(
    command: str,
    model: str = None,
    provider: str = None,
    api_url: str = None,
    api_key: str = None,
    npc: Any = None,
    team: Any = None,
    messages: List[Dict[str, str]] = None,
    images: list = None,
    stream=False,
    context=None,
    shell=False,
    actions: Dict[str, Dict] = None,
    max_iterations: int = 3,
    current_iteration: int = 0,
):
    """This function checks an LLM command and returns sequences of steps with parallel actions."""
    if messages is None:
        messages = []

    if actions is None:
        actions = DEFAULT_ACTION_SPACE.copy()

    return execute_multi_step_plan(
        command=command,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        npc=npc,
        team=team,
        messages=messages,
        images=images,
        stream=stream,
        context=context,
        shell=shell,
        actions=actions,
        max_iterations=max_iterations,
        current_iteration=current_iteration,
        completed_actions=[]  # Initialize empty completed actions
    )

def handle_jinx_call(
    command: str,
    jinx_name: str,
    model: str = None,
    provider: str = None,
    api_url: str = None,
    api_key: str = None,
    messages: List[Dict[str, str]] = None,
    npc: Any = None,
    team: Any = None,
    stream=False,
    shell=False,
    n_attempts=3,
    attempt=0,
    context=None,
    **kwargs
) -> Union[str, Dict[str, Any]]:
    """This function handles a jinx call.
    Args:
        command (str): The command.
        jinx_name (str): The jinx name.
    Keyword Args:
        model (str): The model to use for handling the jinx call.
        provider (str): The provider to use for handling the jinx call.
        messages (List[Dict[str, str]]): The list of messages.
        npc (Any): The NPC object.
    Returns:
        Union[str, Dict[str, Any]]: The result of handling
        the jinx call.

    """
    if npc is None and team is None:
        return f"No jinxs are available. "
    else:

        #print(npc, team)
        #print(team.jinxs_dict, npc.jinxs_dict)
        if jinx_name not in npc.jinxs_dict and jinx_name not in team.jinxs_dict:
            print("not available")

            return  {'output': f"jinx '{jinx_name}' not found in NPC's jinxs_dict.",
                    'messages': messages}
        elif jinx_name in npc.jinxs_dict:
            jinx = npc.jinxs_dict[jinx_name]
        elif jinx_name in team.jinxs_dict:
            jinx = team.jinxs_dict[jinx_name]

        render_markdown(f"jinx found: {jinx.jinx_name}")
        jinja_env = Environment(loader=FileSystemLoader("."), undefined=Undefined)
        example_format = {}
        for inp in jinx.inputs:
            if isinstance(inp, str):
                example_format[inp] = f"<value for {inp}>"
            elif isinstance(inp, dict):
                key = list(inp.keys())[0]
                example_format[key] = f"<value for {key}>"
        
        json_format_str = json.dumps(example_format, indent=4)
        

        prompt = f"""
        The user wants to use the jinx '{jinx_name}' with the following request:
        '{command}'
        Here is the jinx file:
        ```
        {jinx.to_dict()}
        ```

        Please determine the required inputs for the jinx as a JSON object.
        
        They must be exactly as they are named in the jinx.
        For example, if the jinx has three inputs, you should respond with a list of three values that will pass for those args.
        
        If the jinx requires a file path, you must include an absolute path to the file including an extension.
        If the jinx requires code to be generated, you must generate it exactly according to the instructions.
        Your inputs must satisfy the jinx's requirements.

        Return only the JSON object without any markdown formatting.

        The format of the JSON object is:
        {{
        {json_format_str}
        }}
        """

        if npc and hasattr(npc, "shared_context"):
            if npc.shared_context.get("dataframes"):
                context_info = "\nAvailable dataframes:\n"
                for df_name in npc.shared_context["dataframes"].keys():
                    context_info += f"- {df_name}\n"
                prompt += f"""Here is contextual info that may affect your choice: {context_info}
                """
        response = get_llm_response(
            prompt,
            format="json",
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            npc=npc,
            context=context
        )

        try:
            response_text = response.get("response", "{}")
            if isinstance(response_text, str):
                response_text = (
                    response_text.replace("```json", "").replace("```", "").strip()
                )

            # Parse the cleaned response
            if isinstance(response_text, dict):
                input_values = response_text
            else:
                input_values = json.loads(response_text)
            # print(f"Extracted inputs: {input_values}")
        except json.JSONDecodeError as e:
            print(f"Error decoding input values: {e}. Raw response: {response}")
            return f"Error extracting inputs for jinx '{jinx_name}'"
        # Input validation (example):
        required_inputs = jinx.inputs
        missing_inputs = []
        for inp in required_inputs:
            if not isinstance(inp, dict):
                # dicts contain the keywords so its fine if theyre missing from the inputs.
                if inp not in input_values or input_values[inp] == "":
                    missing_inputs.append(inp)
        if len(missing_inputs) > 0:
            # print(f"Missing required inputs for jinx '{jinx_name}': {missing_inputs}")
            if attempt < n_attempts:
                print(f"attempt {attempt+1} to generate inputs failed, trying again")
                print("missing inputs", missing_inputs)
                print("llm response", response)
                print("input values", input_values)
                return handle_jinx_call(
                    command +' . In the previous attempt, the inputs were: ' + str(input_values) + ' and the missing inputs were: ' + str(missing_inputs) +' . Please ensure to not make this mistake again.',
                    jinx_name,
                    model=model,
                    provider=provider,
                    messages=messages,
                    npc=npc,
                    team=team,
                    api_url=api_url,
                    api_key=api_key,
                    stream=stream,
                    attempt=attempt + 1,
                    n_attempts=n_attempts,
                    context=context
                )
            return {
                "output": f"Missing inputs for jinx '{jinx_name}': {missing_inputs}",
                "messages": messages,
            }

        
        render_markdown( "\n".join(['\n - ' + str(key) + ': ' +str(val) for key, val in input_values.items()]))

        try:
            jinx_output = jinx.execute(
                input_values,
                jinja_env,
                npc=npc,
                messages=messages,
            )
            if 'llm_response' in jinx_output and 'messages' in jinx_output:
                if len(jinx_output['llm_response'])>0:                
                    messages = jinx_output['messages']
        except Exception as e:
            print(f"An error occurred while executing the jinx: {e}")
            print(f"trying again, attempt {attempt+1}")
            print('command', command)
            if attempt < n_attempts:
                jinx_output = handle_jinx_call(
                    command,
                    jinx_name,
                    model=model,
                    provider=provider,
                    messages=messages,
                    npc=npc,
                    team=team,
                    api_url=api_url,
                    api_key=api_key,
                    stream=stream,
                    attempt=attempt + 1,
                    n_attempts=n_attempts,
                    context=f""" \n \n \n "jinx failed: {e}  \n \n \n here was the previous attempt: {input_values}""",
                )
            else:
                if shell:
                    user_input = input(
                        "the jinx execution has failed after three tries, can you add more context to help or would you like to run again?"
                    )
                    return handle_jinx_call(
                        command + " " + user_input,
                        jinx_name,
                        model=model,
                        provider=provider,
                        messages=messages,
                        npc=npc,
                        team=team,
                        api_url=api_url,
                        api_key=api_key,
                        stream=stream,
                        attempt=attempt + 1,
                        n_attempts=n_attempts,
                        context=context,
                    )
                else:
                    return {
                        "output": f"Jinx execution failed after {n_attempts} attempts: {e}",
                        "messages": messages,
                    }

        if not stream and messages[-1]['role'] != 'assistant':
            # if the jinx has already added a message to the output from a final prompt we dont want to double that
            
            render_markdown(f""" ## jinx OUTPUT FROM CALLING {jinx_name} \n \n output:{jinx_output['output']}""" )

            
            response = get_llm_response(f"""
                The user had the following request: {command}. 
                Here were the jinx outputs from calling {jinx_name}: {jinx_output}
                
                Given the jinx outputs and the user request, please format a simple answer that 
                provides the answer without requiring the user to carry out any further steps.
                """,
                model=model,
                provider=provider,
                api_url=api_url,
                api_key=api_key,
                npc=npc,
                messages=messages,
                context=context, 
                stream=stream,
            )
            messages = response['messages']
            response = response.get("response", {})
            return {'messages':messages, 'output':response}
        return {'messages': messages, 'output': jinx_output['output']}


def handle_request_input(
    context: str,
    model: str ,
    provider: str 
):
    """
    Analyze text and decide what to request from the user
    """
    prompt = f"""
    Analyze the text:
    {context}
    and determine what additional input is needed.
    Return a JSON object with:
    {{
        "input_needed": boolean,
        "request_reason": string explaining why input is needed,
        "request_prompt": string to show user if input needed
    }}

    Do not include any additional markdown formatting or leading ```json tags. Your response
    must be a valid JSON object.
    """

    response = get_llm_response(
        prompt,
        model=model,
        provider=provider,
        messages=[],
        format="json",
    )

    result = response.get("response", {})
    if isinstance(result, str):
        result = json.loads(result)

    user_input = request_user_input(
        {"reason": result["request_reason"], "prompt": result["request_prompt"]},
    )
    return user_input
def jinx_handler(command, extracted_data, **kwargs):
    return handle_jinx_call(
        command, 
        extracted_data.get('jinx_name'),
        model=kwargs.get('model'),
        provider=kwargs.get('provider'),
        api_url=kwargs.get('api_url'),
        api_key=kwargs.get('api_key'),
        messages=kwargs.get('messages'),
        npc=kwargs.get('npc'),
        team = kwargs.get('team'),
        stream=kwargs.get('stream'),
        shell=kwargs.get('shell'),
        context=kwargs.get('context')
    )

def answer_handler(command, extracted_data, **kwargs):
    print(f"ANSWER HANDLER: cmd='{command}', extracted_data={extracted_data}")
    response =  get_llm_response(
        f"""
        Here is the user question: {command}
        The action chosen by the agent is: answer_question
        The explanation provided by the agent is: {extracted_data.get('explanation', '')}

        Do not needlessly reference the user's files or provided context. Simply provide the answer to the user's question. Avoid
        appearing zany or unnecessarily forthcoming about the fact that you have received such information. You know it
        and the user knows it. there is no need to constantly mention the facts that are aware to both.
        """,
        model=kwargs.get('model'),
        provider=kwargs.get('provider'),
        api_url=kwargs.get('api_url'),
        api_key=kwargs.get('api_key'),
        messages=kwargs.get('messages'),
        npc=kwargs.get('npc'),
        stream=False,
        images=kwargs.get('images')
    )
    
    return response
    
# Define `DEFAULT_ACTION_SPACE`

DEFAULT_ACTION_SPACE = {
    "invoke_jinx": {
        "description": "Invoke a jinx (jinja-template execution script)",
        "handler": jinx_handler,
        "context": lambda npc=None, team=None, **_: (
            """
            Use jinxs when appropriate. For example:

            - If you are asked about something up-to-date or dynamic (e.g., latest exchange rates)
            - If the user asks you to read or edit a file
            - If the user asks for code that should be executed
            - If the user requests to open, search, download or scrape, which involve actual system or web actions
            - If they request a screenshot, audio, or image manipulation
            - Situations requiring file parsing (e.g., CSV or JSON loading)
            - Scripted workflows or pipelines, e.g., generate a chart, fetch data, summarize from source, etc.

            You MUST use a jinx if the request directly refers to a tool the AI cannot handle directly (e.g., 'run', 'open', 'search', etc).

            To invoke a jinx, return the action 'invoke_jinx' along with the jinx specific name. 
            An example for a jinx-specific return would be
            {
                "action": "invoke_jinx",
                "jinx_name": "file_reader",
                "explanation": "Read the contents of <full_filename_path_from_user_request> and <detailed explanation of how to accomplish the problem outlined in the request> ."
            }
            Do not use the jinx names as the action keys.  You must use the action 'invoke_jinx' to invoke a jinx!

            Here are the currently available jinxs:
            """ +
            (
                "\n".join(
                    f"- {name}: {jinx.description}"
                    for name, jinx in getattr(npc, "jinxs_dict", {}).items()
                )
                if npc and getattr(npc, "jinxs_dict", None)
                else "  [No jinxs available for this NPC]"
            ) +  "\n \n \n Team Jinxs: " + (
                "\n".join(
                    f"- {name}: {jinx.description}"
                    for name, jinx in getattr(team, "jinxs_dict", {}).items()
                )
                if team and getattr(team, "jinxs_dict", None)
                else "  [No jinxs available for this team]"
            )
        ),
        "output_keys": {
            "jinx_name": {
                "description": "The name of the jinx to invoke",
                "type": "string"
            }
        }
    },
    "answer_question": {
        "description": "Provide a direct informative answer",
        "handler": answer_handler,
        "context": "For general questions, use existing knowledge",
        "output_keys": {}
    }
}
def plan_multi_step_actions(
   command: str,
   actions: Dict[str, Dict],
   npc: Any = None,
   model: str = None,
   provider: str = None,
   api_url: str = None,
   api_key: str = None,
   context: str = None,
   messages: List[Dict[str, str]] = None,
   team: Any = None
):
    """
    Analyzes the user's command and creates a complete, sequential plan of actions
    by dynamically building a prompt from the provided action space.
    """
    
    # --- Start of the corrected logic ---
    
    # 1. Build the prompt dynamically.
    prompt = f"""
Analyze the user's request: "{command}"

Your task is to create a complete, sequential JSON plan to fulfill the entire request.
Use the following context about available actions and tools to construct the plan.

"""

    # 2. Dynamically add context from the action space, executing lambdas as needed.
    for action_name, action_info in actions.items():
        ctx = action_info.get("context")
        # If the context is a function (our lambda), call it to get the dynamic string.
        if callable(ctx):
            try:
                # Pass the npc object to the lambda so it can access the jinxs_dict.
                ctx = ctx(npc=npc, team=team)
            except Exception as e:
                print(f"[WARN] Failed to render context for action '{action_name}': {e}")
                ctx = None
        
        if ctx:
            prompt += f"\n--- Context for action '{action_name}' ---\n{ctx}\n"

    print(prompt)
    # 3. Add the final instructions.
    prompt += f"""
--- Instructions ---
Based on the user's request and the context provided above, create a plan.
The plan must be a JSON object with a single key, "actions". Each action must include:
- "action": The name of the action to take.
- "explanation": A clear description of the goal for this specific step.

Example Request: "Find out who the current CEO of Microsoft is, then find their biography on Wikipedia"
 
An Example Plan might look like this depending on the available actions:
""" + """
{
  "actions": [
    {
      "action": "<action_name_1>",
      "<action_specific_key_1..>": "<action_specific_value_1>",
      <...> : ...,      
      "explanation": "Identify the current CEO of Microsoft."
    },
    {
      "action": "<action_name_2>",
        "<action_specific_key_1..>": "<action_specific_value_1>",
        "explanation": "Find the <action-specific> information identified in the previous step."
    }
  ]
}

"""+f"""
Now, create the complete plan for the user's query: "{command}"
Respond ONLY with the JSON plan.
"""

    action_response = get_llm_response(
        prompt,
        model=model, provider=provider, api_url=api_url, api_key=api_key,
        npc=npc, format="json", messages=[]
    )

    response_content = action_response.get("response", {})
    if isinstance(response_content, str):
        try:
            response_content = json.loads(response_content)
        except json.JSONDecodeError:
            print(f"Error: Could not parse plan from LLM. Response: {response_content}")
            return []
    
    return response_content.get("actions", [])

def execute_multi_step_plan(
   command: str,
   model: str = None,
   provider: str = None,
   api_url: str = None,
   api_key: str = None,
   npc: Any = None,
   team: Any = None,
   messages: List[Dict[str, str]] = None,
   images: list = None,
   stream=False,
   context=None,
   shell=False,
   actions: Dict[str, Dict] = None,
   max_iterations: int = 3, # No longer used for recursion
   current_iteration: int = 0, # No longer used for recursion
   completed_actions: List = None # No longer used for recursion
):
    """
    Creates a comprehensive plan and executes it sequentially, passing context
    between steps for adaptive behavior.
    """
    
    # 1. Get the complete plan upfront with the corrected call.
    planned_actions = plan_multi_step_actions(
        command=command,
        actions=actions,
        npc=npc,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        context=context,
        messages=messages,
        team=team
    )
    
    if not planned_actions:
        print("Could not generate a multi-step plan. Answering directly.")
        result = answer_handler(command=command, extracted_data={"explanation": "Answering the user's query directly."}, model=model, provider=provider, api_url=api_url, api_key=api_key, messages=messages, npc=npc, images=images)
        return {"messages": result.get('messages', messages), "output": result.get('response')}

    # 2. Execute the plan step-by-step.
    step_outputs = []
    current_messages = messages.copy()
    print(planned_actions)
    for i, action_data in enumerate(planned_actions):
        print(f"--- Executing Step {i + 1} of {len(planned_actions)} ---")
        print(f"Goal: {action_data.get('explanation')}")

        action_name = action_data["action"]
        handler = actions[action_name]["handler"]
        
        step_context = f"Context from previous steps: {json.dumps(step_outputs)}" if step_outputs else ""
        step_command = action_data.get('explanation')

        result = handler(
           command=step_command, 
           extracted_data=action_data,
           model=model,
           provider=provider, 
           api_url=api_url,
           api_key=api_key, 
           messages=current_messages, 
           npc=npc,
           team=team,
           stream=stream, 
           shell=shell, 
           context=step_context, 
           images=images
        )

        action_output = result.get('output') or result.get('response')
        step_outputs.append(action_output)
        current_messages = result.get('messages', current_messages)

    # 3. Compile the final result, passing the original command for context.
    final_output = compile_sequence_results(
       original_command=command,
       outputs=step_outputs,
       model=model, provider=provider, api_url=api_url,
       api_key=api_key, npc=npc
    )
    return {"messages": current_messages, "output": final_output}

def compile_sequence_results(original_command: str, outputs: List[str], model: str = None, provider: str = None, api_url: str = None, api_key: str = None, npc: Any = None) -> str:
    """
    Synthesizes a list of outputs from sequential steps into a single,
    coherent final response, framed as an answer to the original query.
    """
    if not outputs:
        return "The process completed, but produced no output."
    if len(outputs) == 1:
        return outputs[0]
    
    synthesis_prompt = f"""
A user asked the following question:
"{original_command}"

To answer this, the following information was gathered in sequential steps:
{json.dumps(outputs, indent=2)}

Based *directly on the user's original question* and the information gathered, please
provide a single, final, and coherent response. Answer the user's question directly.
Do not mention the steps taken.

Final Synthesized Response:
"""
    
    try:
        response = get_llm_response(
            synthesis_prompt,
            model=model, provider=provider, api_url=api_url,
            api_key=api_key, npc=npc, messages=[]
        )
        synthesized = response.get("response", "")
        if synthesized and synthesized.strip():
            return synthesized
    except Exception as e:
        print(f"Error during final synthesis: {e}")
    
    return "\n\n".join(outputs)
def compile_sequence_results(original_command: str, outputs: List[str], model: str = None, provider: str = None, api_url: str = None, api_key: str = None, npc: Any = None) -> str:
    """
    Synthesizes a list of outputs from sequential steps into a single,
    coherent final response, framed as an answer to the original query.
    """
    
    if not outputs:
        return "The process completed, but produced no output."
    
    if len(outputs) == 1:
        return outputs[0]
    
    # The prompt now includes the user's query, giving the LLM crucial context.
    synthesis_prompt = f"""
A user asked the following question:
"{original_command}"

To answer this, the following information was gathered in sequential steps:
{json.dumps(outputs, indent=2)}

Based *directly on the user's original question* and the information gathered, please
provide a single, final, and coherent response. Answer the user's question directly.
Do not mention the steps taken.

Final Synthesized Response:
"""
    
    try:
        response = get_llm_response(
            synthesis_prompt,
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            npc=npc,
            messages=[],
        )
        
        synthesized = response.get("response", "")
        if synthesized and synthesized.strip():
            return synthesized
    except Exception as e:
        print(f"Error during final synthesis: {e}")
    
    # Fallback to a simple join if synthesis fails.
    return "\n\n".join(outputs)


def should_continue_with_more_actions(
    original_command: str,
    completed_actions: List[Dict[str, Any]],
    current_messages: List[Dict[str, str]],
    model: str = None,
    provider: str = None,
    api_url: str = None,
    api_key: str = None,
    npc: Any = None,
) -> Dict:
    """Decide if more action sequences are needed."""
    
    results_summary = ""
    for idx, action_result in enumerate(completed_actions):
        action_name = action_result.get("action", "Unknown Action")
        output = action_result.get('output', 'No Output')
        output_preview = output[:100] + "..." if isinstance(output, str) and len(output) > 100 else output
        results_summary += f"{idx + 1}. {action_name}: {output_preview}\n"

    prompt = f"""
Original user request: "{original_command}"

This request asks for multiple things. Analyze if ALL parts have been addressed.
Look for keywords like "and then", "use that to", "after that" which indicate multiple tasks.

Completed actions so far:
{results_summary}

For the request "{original_command}", identify:
1. What parts have been completed
2. What parts still need to be done

JSON response:
{{
    "needs_more_actions": true/false,
    "reasoning": "explain what's been done and what's still needed",
    "next_focus": "if more actions needed, what specific task should be done next"
}}
"""

    response = get_llm_response(
        prompt,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        npc=npc,
        format="json",
        messages=[],
        context=None,
    )
    
    response_dict = response.get("response", {})
    if not isinstance(response_dict, dict):
        return {"needs_more_actions": False, "reasoning": "Error", "next_focus": ""}
        
    return response_dict