# vibetuner

**Core Python framework and blessed dependencies for Vibetuner projects**

This package provides the complete Python framework and curated dependency set for building modern web applications with Vibetuner. It includes everything from FastAPI and MongoDB integration to authentication, background jobs, and CLI tools.

## What is Vibetuner?

Vibetuner is a production-ready scaffolding tool for FastAPI + MongoDB + HTMX web applications. This package (`vibetuner`) is the Python component that provides:

- Complete web application framework built on FastAPI
- MongoDB integration with Beanie ODM
- OAuth and magic link authentication out of the box
- Background job processing with Redis + Streaq
- CLI framework with Typer
- Email services, blob storage, and more

**This package is designed to be used within projects generated by the Vibetuner scaffolding tool.** For standalone use, you'll need to set up the project structure manually.

## Installation

```bash
# In a Vibetuner-generated project (automatic)
uv sync

# Add to an existing project
uv add vibetuner

# With development dependencies
uv add vibetuner[dev]
```

## Quick Start

The recommended way to use Vibetuner is via the scaffolding tool:

```bash
# Create a new project with all the framework code
uvx vibetuner scaffold new my-project
cd my-project
just dev
```

This will generate a complete project with:
- Pre-configured FastAPI application
- Authentication system (OAuth + magic links)
- MongoDB models and configuration
- Frontend templates and asset pipeline
- Docker setup for development and production
- CLI commands and background job infrastructure

## What's Included

### Core Framework (`src/vibetuner/`)

- **`frontend/`**: FastAPI app, routes, middleware, auth
- **`models/`**: User, OAuth, email verification, blob storage models
- **`services/`**: Email (SES), blob storage (S3)
- **`tasks/`**: Background job infrastructure
- **`cli/`**: CLI framework with scaffold, run commands
- **`config.py`**: Pydantic settings management
- **`mongo.py`**: MongoDB/Beanie setup
- **`logging.py`**: Structured logging configuration

### Blessed Dependencies

- **FastAPI** (0.121+): Modern, fast web framework
- **Beanie**: Async MongoDB ODM with Pydantic
- **Authlib**: OAuth 1.0/2.0 client
- **Granian**: High-performance ASGI server
- **Redis** + **Streaq**: Background task processing
- **Typer**: CLI framework
- **Rich**: Beautiful terminal output
- **Loguru**: Structured logging
- **Pydantic**: Data validation and settings

See [pyproject.toml](./pyproject.toml) for the complete dependency list.

## CLI Tools

When installed, provides the `vibetuner` command:

```bash
# Create new project from template
vibetuner scaffold new my-project
vibetuner scaffold new my-project --defaults

# Update existing project
vibetuner scaffold update

# Run development server (in generated projects)
vibetuner run dev frontend
vibetuner run dev worker

# Run production server (in generated projects)
vibetuner run prod frontend
vibetuner run prod worker
```

## Development Dependencies

The `[dev]` extra includes all tools needed for development:

- **Ruff**: Fast linting and formatting
- **djlint**: Template linting
- **Babel**: i18n message extraction
- **pre-commit**: Git hooks
- **Type stubs**: For aioboto3, authlib, PyYAML
- And more...

## Usage in Generated Projects

In a Vibetuner-generated project, import from `vibetuner`:

```python
# Use core models
from vibetuner.models import UserModel, OAuthAccountModel

# Use services
from vibetuner.services.email import send_email

# Use configuration
from vibetuner.config import settings

# Extend core routes
from vibetuner.frontend import app

# Add your routes
@app.get("/api/hello")
async def hello():
    return {"message": "Hello World"}
```

## Documentation

For complete documentation, guides, and examples, see the main Vibetuner repository:

**📖 [Vibetuner Documentation](https://github.com/alltuner/vibetuner#readme)**

## Package Ecosystem

Vibetuner consists of three packages that work together:

1. **vibetuner** (this package): Python framework and dependencies
2. **[@alltuner/vibetuner](https://www.npmjs.com/package/@alltuner/vibetuner)**: JavaScript/CSS build dependencies
3. **Scaffolding template**: Copier template for project generation

All three are version-locked and tested together to ensure compatibility.

## Contributing

Contributions welcome! See the main repository for contribution guidelines:

**🤝 [Contributing to Vibetuner](https://github.com/alltuner/vibetuner/blob/main/CONTRIBUTING.md)**

## License

MIT License - Copyright (c) 2025 All Tuner Labs, S.L.

See [LICENSE](https://github.com/alltuner/vibetuner/blob/main/LICENSE) for details.

## Links

- **Main Repository**: https://github.com/alltuner/vibetuner
- **Documentation**: https://github.com/alltuner/vibetuner#readme
- **Issues**: https://github.com/alltuner/vibetuner/issues
- **PyPI**: https://pypi.org/project/vibetuner/
- **npm Package**: https://www.npmjs.com/package/@alltuner/vibetuner
