"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveServiceName = resolveServiceName;
exports.extractResolvedServiceName = extractResolvedServiceName;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const variants_1 = require("./variants");
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
/** @internal */
const SERVICE_NAME_LOOKUP = Object.fromEntries([
    ...Object.keys(assets_1.AwsAsset.Services).flatMap((key) => {
        return [
            // constant self lookup
            [key, key],
            [(0, utils_1.normalizeComparisonString)(key), key],
        ];
    }),
    ...Object.entries(variants_1.NORMALIZED_SERVICE_VARIANTS).flatMap(([key, variants]) => {
        if (variants == null)
            return [];
        return variants.map((variant) => [(0, utils_1.normalizeComparisonString)(variant), key]);
    }),
]);
/** @internal */
function resolveServiceName(value) {
    const comparableValue = (0, utils_1.normalizeComparisonString)(value);
    // constant lookup first for perf
    if (comparableValue in SERVICE_NAME_LOOKUP) {
        return SERVICE_NAME_LOOKUP[comparableValue];
    }
    // attempt know mappings
    if (comparableValue.startsWith("media") &&
        SERVICE_NAME_LOOKUP[`elemental${comparableValue}`]) {
        const serviceName = SERVICE_NAME_LOOKUP[`elemental${comparableValue}`];
        SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
        return serviceName;
    }
    // prefixed lookup (WAFRegion => WAF, CodeGuruProfile => CodeGuru)
    for (const serviceName of Object.keys(assets_1.AwsAsset.Services)) {
        if (comparableValue.startsWith((0, utils_1.normalizeComparisonString)(serviceName))) {
            // add to lookup for constant lookkup in subsequent calls;
            SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
            return serviceName;
        }
    }
    throw new Error(`Failed to resolve serviceName ${value} (${comparableValue})`);
}
/**
 * Extract service name parts from value that is or contains a service name or variant of service name.
 * @internal
 */
function extractResolvedServiceName(value) {
    const parts = (0, utils_1.normalizeIdentifier)(value).split("_");
    for (let i = parts.length; i > 0; i--) {
        try {
            const match = parts.slice(0, i).join("_");
            const serviceName = resolveServiceName(match);
            let rest = value.replace(match, "");
            if (rest.length === 0) {
                rest = undefined;
            }
            const resolvedValue = `${serviceName}${rest}`;
            return { serviceName, match, rest, resolvedValue };
        }
        catch {
            continue;
        }
    }
    return;
}
//# sourceMappingURL=data:application/json;base64,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