"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AWS_URL_PATTERN = void 0;
exports.normalizeIdentifier = normalizeIdentifier;
exports.normalizeComparisonString = normalizeComparisonString;
exports.parseAwsUrl = parseAwsUrl;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
/**
 * Normalize string value to standard identifier form.
 *
 * Removes extraneous prefixes (Amazon, AWS), replaces all non-alphanumerics with underscore (_),
 * and converts to lower-case.
 * @param value - Value to normalize
 * @returns Returns normalized identifier string.
 */
function normalizeIdentifier(value) {
    value = value.replace(/^(Amazon|AWS)-?/i, ""); // remove prefix
    value = value.replace(/[^A-Za-z0-9]/g, "_"); // only alphanum + _
    value = value.replace(/_+/g, "_"); // de-dupe _
    return value.toLowerCase();
}
const ComparisonDictionary = {
    acl: "access_control_list",
    eni: "elastic_network_interface",
    eip: "elastic_ip_address",
};
/**
 * Normalizes string value for consistent comparison with variants between systems.
 * @param value - The value to normalize for comparison
 */
function normalizeComparisonString(value) {
    value = value.replace(/\.(png|svg)$/, "");
    value = normalizeIdentifier(value);
    Object.entries(ComparisonDictionary).forEach(([_key, _value]) => {
        _value = _value.replace(new RegExp(_key, "ig"), _value);
    });
    value = value.replace(/[_-]+(amazon|aws)[_-]+/gi, "");
    value = value.replace(/_(for|on|and|the|of|&)_/gi, "");
    value = value.replace(/v\d+/g, "");
    value = value.replace(/_/g, "");
    return value;
}
/**
 * RegEx pattern for https://aws.amazon.com based urls.
 */
exports.AWS_URL_PATTERN = /^https:\/\/(?<domain>(?:(?:docs)\.)?aws\.amazon\.com)\/(?<service>[^\/]+)(?:\/(?<resource>[^#]+))?\/?(?:#(?<dataType>.+))?$/i;
/** Parses https://aws.amazon.com based url into common definition. */
function parseAwsUrl(url) {
    url = url.replace(/\/(index\.html.*)?$/, "");
    const groups = url.match(exports.AWS_URL_PATTERN)
        ?.groups;
    if (groups) {
        const code = normalizeIdentifier(`${groups.service}_${groups.resource || ""}_${groups.dataType || ""}`);
        return {
            ...groups,
            code,
        };
    }
    throw new Error(`Failed to parse aws url: ${url}`);
}
//# sourceMappingURL=data:application/json;base64,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