"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Diagram = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const Dot = require("ts-graphviz");
const theme_1 = require("../theme");
/**
 * Diagram class defines the root {@link Dot.Digraph} of the diagram
 * @internal
 */
class Diagram extends Dot.Digraph {
    /** @internal */
    constructor(label, imagepath) {
        const { graph, node, edge } = theme_1.GraphTheme.instance;
        super("ROOT", {
            label,
            imagepath,
        });
        this._trackedImages = new Set();
        this.imagepath = imagepath;
        this.apply(graph);
        if (theme_1.GraphTheme.instance.rendering.layout === "vertical") {
            this.set("rankdir", "LR");
        }
        // NB: do not apply "subgraph" attributes as there seems to be bug where it overrides the diagram attributes
        // The subgraph class will apply them directly so not a concern
        this.attributes.node.apply(node);
        this.attributes.edge.apply(edge);
    }
    /**
     * Track image used in the graph for downstream tooling support integration.
     *
     * @see [dot-wasm options](https://hpcc-systems.github.io/hpcc-js-wasm/classes/graphviz.Graphviz.html#layout)
     * @param image Image to track
     */
    trackImage(image) {
        this._trackedImages.add(image);
    }
    /** Get list of all tracked images */
    getTrackedImages() {
        return Array.from(this._trackedImages);
    }
    /**
     * Converts the diagram to **dot** string format
     * @see {@link Dot.toDot}
     */
    toDot() {
        return Dot.toDot(this);
    }
}
exports.Diagram = Diagram;
//# sourceMappingURL=data:application/json;base64,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