"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphTheme = void 0;
exports.getBaseTheme = getBaseTheme;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../../aws-arch");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const theme_1 = require("./theme");
const fonts_1 = require("../../fonts");
const DEFAULT_RENDERING = {
    resourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    resourceIconMax: theme_1.GraphThemeRenderingIconTarget.CATEGORY,
    cfnResourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    cfnResourceIconMax: theme_1.GraphThemeRenderingIconTarget.SERVICE,
};
/** GraphTheme class is the implementation of diagram graph based theme */
class GraphTheme {
    /** Initializes the theme based on {@link aws_arch.Theme} definition */
    static init(config) {
        if (config == null || typeof config === "string") {
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(config)));
        }
        else if (Object.keys(config).length <= 2) {
            const { theme, rendering } = config;
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(theme), rendering));
        }
        else {
            this._instance = new GraphTheme(config);
        }
        return this.instance;
    }
    /** Get the current singleton instance for the theme being utilized for the diagram */
    static get instance() {
        if (this._instance == null) {
            throw new Error("Must init GraphTheme before requesting instance");
        }
        return this._instance;
    }
    /** @internal */
    constructor(theme) {
        this.rendering = theme.rendering;
        this.graph = theme.graph;
        this.subgraph = theme.subgraph;
        this.cluster = theme.cluster;
        this.node = theme.node;
        this.edge = theme.edge;
        this.cloud = theme.cloud;
        this.stage = theme.stage;
        this.stack = theme.stack;
        this.nestedStack = theme.nestedStack;
        this.imageNode = theme.imageNode;
        this.cfnResourceNode = theme.cfnResourceNode;
        this.resourceNode = theme.resourceNode;
        this.childLink = theme.childLink;
        this.referenceLink = theme.referenceLink;
        this.dependencyLink = theme.dependencyLink;
        this.awsTheme = theme.awsTheme;
    }
}
exports.GraphTheme = GraphTheme;
/** Get the base theme */
function getBaseTheme(rendering) {
    return cloneDeep({
        rendering: {
            ...DEFAULT_RENDERING,
            ...rendering,
        },
        graph: GRAPH_ATTRIBUTES,
        subgraph: SUBGRAPH_ATTRIBUTES,
        cluster: CLUSTER_ATTRIBUTES,
        node: NODE_ATTRIBUTES,
        edge: EDGE_ATTRIBUTES,
        cloud: CLOUD_ATTRIBUTES,
        stage: STAGE_ATTRIBUTES,
        stack: STACK_ATTRIBUTES,
        nestedStack: NESTED_STACK_ATTRIBUTES,
        imageNode: IMAGE_NODE_ATTRIBUTES,
        cfnResourceNode: CFN_RESOURCE_NODE_ATTRIBUTES,
        resourceNode: RESOURCE_NODE_ATTRIBUTES,
        childLink: CHILD_LINK_ATTRIBUTES,
        referenceLink: REFERENCE_LINK_ATTRIBUTES,
        dependencyLink: DEPENDENCY_LINK_ATTRIBUTES,
    });
}
/** Generate {@link IGraphTheme} from {@link aws_arch.Theme} */
function generateGraphThemeFromAwsTheme(awsTheme, rendering) {
    const theme = getBaseTheme(rendering);
    Object.assign(theme, {
        awsTheme,
    });
    Object.assign(theme.graph, {
        bgcolor: awsTheme.backgrounds.base,
        fontcolor: awsTheme.text.default,
    });
    Object.assign(theme.node, {
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.edge, awsArrowToEdgeAtts(awsTheme.arrows.default));
    // containers
    Object.assign(theme.cloud, awsGroupToSubgraphAtts(awsTheme.groups.cloud));
    Object.assign(theme.stage, awsGroupToSubgraphAtts(awsTheme.groups.awsAccount, true));
    Object.assign(theme.stack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    Object.assign(theme.nestedStack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    // nodes
    Object.assign(theme.cfnResourceNode, {
        color: awsTheme.text.secondary,
        fontcolor: awsTheme.text.secondary,
    });
    Object.assign(theme.resourceNode, {
        color: awsTheme.text.primary,
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.childLink, awsArrowToEdgeAtts(awsTheme.arrows.child));
    Object.assign(theme.referenceLink, awsArrowToEdgeAtts(awsTheme.arrows.reference));
    Object.assign(theme.dependencyLink, awsArrowToEdgeAtts(awsTheme.arrows.dependency));
    return theme;
}
/** Base graph attributes */
const GRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    center: true,
    compound: true,
    concentrate: true,
    dpi: 300,
    fontcolor: "#222222",
    fontnames: "ps",
    fontsize: 14,
    forcelabels: true,
    labelloc: "tc",
    nodesep: 0.8,
    pad: "%2,%1",
    // @ts-ignore
    rankdir: "TB",
    ranksep: 0.75,
    ratio: "compress",
    remincross: true,
    size: "%1024,%1024!",
    splines: "ortho",
    stylesheet: fonts_1.FONT_STYLESHEET,
};
/** Base subgraph attributes */
const SUBGRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,solid",
};
/** Base cluster attributes */
const CLUSTER_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,dashed",
};
/** Base cloud container attributes */
const CLOUD_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD,
    style: "solid",
};
/** Base stage attributes */
const STAGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD_ITALIC,
    style: "dashed",
    margin: 6,
    rank: "same",
};
/** Base stack attributes */
const STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    style: "solid,bold,filled",
    fillcolor: "#5A6B861A", // 10%
    margin: 10,
};
/** Base nested stack attributes */
const NESTED_STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    style: "solid,filled",
    fillcolor: "#5A6B861A", // 10%
    margin: 6,
};
/** Base node attributes */
const NODE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    shape: "box",
    style: "solid",
    fixedsize: false,
    width: 0.25,
    height: 0.25,
    labelloc: "c",
    imagescale: true,
    fontsize: 10,
    penwidth: 0,
};
/** Base image based node attributes */
const IMAGE_NODE_ATTRIBUTES = {
    shape: "box",
    style: "solid,rounded",
    fixedsize: true,
    width: 1,
    height: 1,
    labelloc: "b",
    imagescale: true,
    imagepos: "tc",
    fillcolor: "transparent",
};
/** Base cfn resource node attributes */
const CFN_RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    style: "solid,rounded",
    color: "#999999",
    fontcolor: "#999999",
};
/** Base resource node attributes */
const RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    color: "#666666",
    fontcolor: "#666666",
};
/** Base edge attributes */
const EDGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    dir: "both",
    color: "#545B64",
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "dot",
    arrowsize: 0.5,
    fontsize: 9,
    style: "solid",
};
/** Base child link attributes */
const CHILD_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 1,
    style: "solid",
};
/** Base reference link attributes */
const REFERENCE_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 0.75,
    style: "solid",
};
/** Base dependency link attributes */
const DEPENDENCY_LINK_ATTRIBUTES = {
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "odot",
    arrowsize: 0.75,
    style: "dotted",
};
/** Convert {@link aws_arch.GroupFormat} to {@link Dot.SubgraphAttributesObject} */
function awsGroupToSubgraphAtts(group, ignoreStyle = false) {
    const attributes = {};
    if (!ignoreStyle) {
        if (group.borderStyle === "none") {
            attributes.style = "solid";
            attributes.penwidth = 0;
        }
        else {
            attributes.style = group.borderStyle;
        }
    }
    attributes.color = group.color || "transparent";
    attributes.bgcolor = group.bgcolor || "transparent";
    attributes.fillcolor = group.bgcolor || "transparent";
    attributes.pencolor = (group.borderColor || "transparent");
    if (group.color && group.color !== "transparent") {
        attributes.fontcolor = group.color;
    }
    return attributes;
}
/** Convert {@link aws_arch.ArrowFormat} to {@link Dot.EdgeAttributesObject} */
function awsArrowToEdgeAtts(arrow) {
    return {
        color: arrow.color,
        fontcolor: arrow.color,
        arrowtail: arrow.tail,
        arrowhead: arrow.head,
        penwidth: arrow.width,
        style: arrow.style,
    };
}
//# sourceMappingURL=data:application/json;base64,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