"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.focusFilter = focusFilter;
exports.nonExtraneousFilterPreset = nonExtraneousFilterPreset;
exports.compactFilterPreset = compactFilterPreset;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const filters_1 = require("./filters");
/**
 * Focus the graph on a specific node.
 * @throws Error if store is not filterable
 * @destructive
 */
function focusFilter(store, plan) {
    filters_1.Filters.verifyFilterable(store);
    if (plan.focus == null)
        return; // noop
    let focusedNode;
    let hoist = true;
    const { filter: _filter, noHoist: _noHoist } = plan.focus;
    focusedNode = _filter.filter(store);
    hoist = !_noHoist;
    if (focusedNode === store.root)
        return; // noop
    const ancestors = focusedNode.scopes.slice();
    // remove the actual store.root from ancestors (we can't destroy that)
    const rootAncestor = ancestors.shift();
    if (rootAncestor !== store.root) {
        throw new Error(`${focusedNode} is not within the store root graph: it has root of ${rootAncestor}`);
    }
    if (hoist) {
        // Move focused node as direct child of store root and prune all other ancestors
        focusedNode.mutateHoist(store.root);
        // Only need to destroy to first non-root ancestor to prune the ancestral tree
        if (ancestors.length) {
            ancestors[0].mutateDestroy();
        }
        // prune all other root children (unless preserved)
        store.root.children.forEach((child) => {
            if (child !== focusedNode) {
                child.mutateDestroy();
            }
        });
    }
    else {
        // keep the focused node in place, but prune non-direct ancestor chain nodes
        // the direct ancestor chain is only the nodes scopes
        ancestors.reverse().forEach((ancestor) => {
            ancestor.siblings.forEach((ancestorSibling) => {
                ancestorSibling.mutateDestroy();
            });
        });
        // prune all planRoot siblings
        focusedNode.siblings.forEach((sibling) => {
            sibling.mutateDestroy();
        });
    }
}
/**
 * Preset that performs {@link Filters.pruneExtraneous} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function nonExtraneousFilterPreset(store) {
    return filters_1.Filters.pruneExtraneous().filter(store);
}
/**
 * Preset that performs {@link Filters.compact} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function compactFilterPreset(store) {
    filters_1.Filters.compact().filter(store);
}
//# sourceMappingURL=data:application/json;base64,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