"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructurePyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Python Project.
 */
class InfrastructurePyProject extends awscdk_1.AwsCdkPythonApp {
    constructor(options) {
        const moduleName = options.moduleName ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            poetry: true,
            moduleName,
            appEntrypoint: "main.py",
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/python/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        ["pytest@^7", "syrupy@^4"].forEach((devDep) => this.addDevDependency(devDep));
        ["aws_pdk@^0", "cdk_nag@^2", "python@^3.9"].forEach((dep) => this.addDependency(dep));
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/python/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/python/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.python) {
                throw new Error("Cannot pass in a Type Safe Api without Python Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addPythonPoetryDependency(tsApi.infrastructure.python);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            allowSignup: options.allowSignup ?? false,
            moduleName,
            typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, typeSafeApis),
            stages: options.stages || [],
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, csWebsite.typeSafeApis),
                };
            }),
        };
        const tstDir = "tests";
        options.sample !== false &&
            this.emitSampleFiles(srcDir, [this.moduleName], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, [tstDir], mustacheConfig);
        this.testTask.reset(`poetry run pytest ${tstDir}/ \${CI:-'--snapshot-update'}`);
    }
    generateTypeSafeMustacheConfig(moduleName, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.python?.moduleName,
                moduleName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("api.py.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.py.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...(f.name !== "main.py.mustache" ? pathPrefixes : []), // emit at the root so package imports work correctly :(
                f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructurePyProject = InfrastructurePyProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructurePyProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructurePyProject", version: "0.25.8" };
//# sourceMappingURL=data:application/json;base64,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