"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const javascript_1 = require("projen/lib/javascript");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureTsProject extends awscdk_1.AwsCdkTypeScriptApp {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            name: options.name,
            sampleCode: false,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/typescript/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.addDeps("@aws/pdk", "cdk-nag");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const typeSafeWebSocketApis = options.typeSafeWebSocketApis ?? [];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        [...typeSafeApis, ...typeSafeWebSocketApis].forEach((tsApi) => {
            if (!tsApi.infrastructure.typescript) {
                throw new Error("Cannot pass in a Type Safe Api without Typescript Infrastructure configured!");
            }
            this.addDeps(`${tsApi.infrastructure.typescript?.package.packageName}@${tsApi.infrastructure.typescript?.package.manifest.version}`);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            this.addDevDeps(`${csWebsite.package.packageName}@${csWebsite.package.manifest.version}`);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            allowSignup: options.allowSignup ?? false,
            typeSafeApis: this.generateTypeSafeMustacheConfig(typeSafeApis),
            typeSafeWebSocketApis: this.generateTypeSafeMustacheConfig(typeSafeWebSocketApis),
            stages: options.stages || [],
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(csWebsite.typeSafeApis),
                    typeSafeWebSocketApis: this.generateTypeSafeMustacheConfig(csWebsite.typeSafeWebSocketApis),
                };
            }),
        };
        options.sampleCode !== false &&
            this.emitSampleFiles(srcDir, ["src"], mustacheConfig);
        options.sampleCode !== false &&
            this.emitSampleFiles(testDir, ["test"], mustacheConfig);
        const eslintTask = this.tasks.tryFind("eslint");
        this.testTask.reset();
        this.testTask.exec("jest --passWithNoTests ${CI:-'--updateSnapshot'}");
        eslintTask && this.testTask.spawn(eslintTask);
    }
    generateTypeSafeMustacheConfig(typeSafeApis) {
        return (typeSafeApis ?? []).map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.typescript?.package.packageName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("websocketapi.ts.mustache")) {
                mustacheConfig.typeSafeWebSocketApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("api.ts.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.ts.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                });
            }
        });
    }
}
exports.InfrastructureTsProject = InfrastructureTsProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureTsProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureTsProject", version: "0.25.8" };
//# sourceMappingURL=data:application/json;base64,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