import { Component, JsonFile, Project } from "projen";
import { JavaProject } from "projen/lib/java";
import { PythonProject } from "projen/lib/python";
import { Obj } from "projen/lib/util";
import { Nx } from "../../nx-types";
/**
 * Component which manages the project specific NX Config and is added to all NXMonorepo subprojects.
 * @experimental
 */
export declare class NxProject extends Component {
    /**
     * Retrieves an instance of NXProject if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project: Project): NxProject | undefined;
    /**
     * Retrieves an instance of NXProject if one is associated to the given project,
     * otherwise created a NXProject instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project: Project): NxProject;
    /**
     * Raw json file
     *
     * **Attention:** any overrides applied here will not be visible
     * in the properties and only included in final synthesized output,
     * and likely to override native handling.
     * @advanced
     */
    readonly file: JsonFile;
    /**
     * Named inputs
     * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
     */
    namedInputs: Obj<any>;
    /**
     * Targets configuration
     * @see https://nx.dev/reference/project-configuration
     */
    targets: Obj<any>;
    /**
     * Project tag annotations
     *
     * @see https://nx.dev/reference/project-configuration#tags
     */
    tags: string[];
    /**
     * Implicit dependencies
     *
     * @see https://nx.dev/reference/project-configuration#implicitdependencies
     */
    implicitDependencies: string[];
    /**
     * Explicit list of scripts for Nx to include.
     * @see https://nx.dev/reference/project-configuration#ignoring-package.json-scripts
     */
    includedScripts: string[];
    constructor(project: Project);
    /**
     * Automatically infer targets based on project type.
     * @experimental
     */
    inferTargets(): void;
    /** Merge configuration into existing config */
    merge(config: Nx.ProjectConfig): void;
    /** Add tag */
    addTag(...tags: string[]): void;
    /**
     * Adds an implicit dependency between the dependant (this project) and dependee.
     *
     * @param dependee project to add the implicit dependency on.
     */
    addImplicitDependency(...dependee: (Project | string)[]): void;
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependee: JavaProject): void;
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependee: PythonProject): void;
    /** Set `namedInputs` helper */
    setNamedInput(name: string, inputs: string[]): void;
    /** @internal */
    protected _getTargetDefaults(name: string): Nx.IProjectTarget | {};
    /** Set `targets` helper */
    setTarget(name: string, target: Nx.IProjectTarget, includeDefaults?: boolean | string): void;
    /**
     * Add input and output files to build target
     * @param inputs Input files
     * @param outputs Output files
     */
    addBuildTargetFiles(inputs?: (string | Nx.IInput)[], outputs?: string[]): void;
    /** @interface */
    synthesize(): void;
}
