"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inferBuildTarget = inferBuildTarget;
const cdk_1 = require("projen/lib/cdk");
const java_1 = require("projen/lib/java");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const utils_1 = require("../../utils");
/**
 * Defines a fileset for target inputs and outputs.
 * @experimental
 * @internal
 */
class TargetFileset {
    static File(fileset, isWorkspaceRoot = false) {
        return new TargetFileset(fileset, false, isWorkspaceRoot);
    }
    static Directory(fileset, isWorkspaceRoot = false) {
        return new TargetFileset(fileset, true, isWorkspaceRoot);
    }
    static resolveInputs(values) {
        return values.map((v) => (typeof v === "string" ? v : v.filesetStarIfDir));
    }
    static resolveOutputs(values) {
        return values.map((v) => (typeof v === "string" ? v : v.fileset));
    }
    static outputsToInputs(values) {
        return values.map((v) => typeof v === "string" ? v : v.inverse().filesetStarIfDir);
    }
    constructor(fileset, isDirectory, isWorkspaceRoot) {
        this._fileset = fileset;
        this.isDirectory = isDirectory;
        this.isExclude = this._fileset.startsWith("!");
        this.isWorkspaceRoot = isWorkspaceRoot;
    }
    _ensureWildcards(fileset) {
        if (fileset.endsWith("*")) {
            return fileset;
        }
        if (fileset.endsWith("/")) {
            return fileset + "**/*";
        }
        return fileset + "/**/*";
    }
    get fileset() {
        let fileset = this.isExclude ? this._fileset.substring(1) : this._fileset;
        if (fileset.startsWith("/")) {
            fileset = fileset.substring(1);
        }
        if (this.isWorkspaceRoot) {
            return `${this.isExclude ? "!" : ""}{workspaceRoot}/${fileset}`;
        }
        return `${this.isExclude ? "!" : ""}{projectRoot}/${fileset}`;
    }
    inverse() {
        if (this.isExclude) {
            return new TargetFileset(this._fileset.substring(1), this.isDirectory, this.isWorkspaceRoot);
        }
        return new TargetFileset("!" + this._fileset, this.isDirectory, this.isWorkspaceRoot);
    }
    get filesetStarIfDir() {
        if (this.isDirectory) {
            return this._ensureWildcards(this.fileset);
        }
        return this.fileset;
    }
}
/**
 * Infer nx target values for build task
 * @experimental
 * @internal
 */
function inferBuildTarget(project, options) {
    const { inputs = [], outputs = [] } = _inferBuildTargetIO(project);
    if (options?.excludeOutputs !== false) {
        inputs.push(...TargetFileset.outputsToInputs(outputs));
    }
    let dependsOn = ["^build"];
    if (options?.dependsOn != null && options.dependsOn !== true) {
        if (options.dependsOn === false) {
            dependsOn = undefined;
        }
        else {
            dependsOn = options.dependsOn;
        }
    }
    if (outputs.length === 0) {
        return undefined;
    }
    return {
        inputs: TargetFileset.resolveInputs(inputs),
        outputs: TargetFileset.resolveOutputs(outputs),
        dependsOn,
    };
}
/** @internal */
function _inferBuildTargetIO(project) {
    const inputs = [];
    const outputs = [];
    let includeDefaultInputs = true;
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, cdk_1.JsiiProject)) {
        outputs.push(TargetFileset.File(".jsii"), TargetFileset.Directory(project.libdir), TargetFileset.Directory(project.artifactsDirectory));
    }
    if (javascript_1.Jest.of(project)) {
        outputs.push(TargetFileset.Directory("coverage"), TargetFileset.Directory("test-reports"));
    }
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, python_1.PythonProject)) {
        inputs.push(TargetFileset.Directory("!.env"), TargetFileset.Directory("!.pytest_cache"));
    }
    if (utils_1.ProjectUtils.isNamedInstanceOf(project, java_1.JavaProject)) {
        inputs.push(TargetFileset.File("!.classpath"), TargetFileset.File("!.project"), TargetFileset.File("!.settings"));
        outputs.push(TargetFileset.Directory("target"), TargetFileset.Directory("dist/java"));
    }
    if (includeDefaultInputs) {
        inputs.unshift("default", "^default");
    }
    return {
        inputs,
        outputs,
    };
}
//# sourceMappingURL=data:application/json;base64,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