"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-ts
 */
class MonorepoTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const defaultReleaseBranch = options.defaultReleaseBranch ?? "main";
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            projenCommand: options.packageManager
                ? utils_1.NodePackageUtils.command.projen(options.packageManager)
                : undefined,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            sampleCode: false, // root should never have sample code,
            gitignore: [".tmp", ...(options.gitignore ?? [])],
            defaultReleaseBranch,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts", ".projenrc.ts"],
            },
            peerDeps: ["nx@^19", ...(options.peerDeps || [])],
            devDeps: ["nx@^19", "@aws/pdk@^0", ...(options.devDeps || [])],
            deps: [
                "aws-cdk-lib",
                "cdk-nag",
                "@aws-cdk/aws-cognito-identitypool-alpha@latest",
                ...(options.deps || []),
            ],
            projenVersion: options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION,
        });
        this.subNodeProjectResolves = [];
        this.projenVersion = options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION;
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch,
            licenseOptions: options.licenseOptions,
        });
        this._options = options;
        // engines
        this.package.addEngine("node", ">=16");
        this.package.setScript("install:ci", !this.ejected
            ? utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen install:ci")
            : "scripts/run-task install:ci");
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.BUN: {
                this.package.addEngine("bun", ">=1");
                break;
            }
            case javascript_1.NodePackageManager.PNPM: {
                // https://pnpm.io/package_json
                // https://github.com/pnpm/pnpm/releases/tag/v8.0.0
                this.package.addEngine("pnpm", ">=8 <9");
                break;
            }
            case javascript_1.NodePackageManager.YARN_CLASSIC:
            case javascript_1.NodePackageManager.YARN: {
                this.package.addEngine("yarn", ">=1 <2");
                break;
            }
            case javascript_1.NodePackageManager.YARN_BERRY:
            case javascript_1.NodePackageManager.YARN2: {
                this.package.addEngine("yarn", ">=2");
                // Yarn Berry cannot call yarn exec without an install first! Use NPX instead.
                this.package.setScript("install:ci", !this.ejected
                    ? utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, "projen install:ci")
                    : "scripts/run-task install:ci");
                this.package.setScript("install", !this.ejected
                    ? utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, "projen install")
                    : "scripts/run-task install");
                this.gitignore.addPatterns(".yarn/*", ".pnp.cjs", "!.yarn/releases", "!.yarn/plugins");
                break;
            }
            case javascript_1.NodePackageManager.NPM: {
                // Allow older versions of peer deps to resolv compatibility issues
                this.tasks.tryFind("install")?.reset("npm install --legacy-peer-deps");
                this.tasks.tryFind("install:ci")?.reset("npm ci --legacy-peer-deps");
                break;
            }
        }
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        !this.ejected &&
            this.package.setScript("synth-workspace", utils_1.NodePackageUtils.command.projen(this.package.packageManager));
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
                target: "pre-compile",
            });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
                target: "compile",
            });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this.nxConfigurator._overrideNxBuildTask(this.testTask, {
                target: "test",
            });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            // The Projenrc component of TypeScriptProject resets the eslint task as part of preSynthesize which would undo
            // our changes, so we disable further resets.
            this.nxConfigurator._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" }, { disableReset: true });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
                target: "package",
            });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this.nxConfigurator._overrideNxBuildTask("prepare", {
                target: "prepare",
            });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this.nxConfigurator._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2", "wrap-ansi@^7.0.0");
        if (this.package.packageManager !== javascript_1.NodePackageManager.BUN) {
            this.package.addPackageResolutions("@zkochan/js-yaml@npm:js-yaml@4.1.0");
        }
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.sortedSubProjects
            .filter((s) => utils_1.ProjectUtils.isNamedInstanceOf(s, javascript_1.NodeProject))
            .map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /** Get consistently sorted list of subprojects */
    get sortedSubProjects() {
        return this.subprojects
            .slice()
            .sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    linkLocalWorkspaceBins() {
        const bins = [];
        this.subprojects.forEach((subProject) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(subProject, javascript_1.NodeProject) &&
                subProject.name !== "@aws/pdk") {
                const pkgBins = subProject.package.manifest.bin() || {};
                bins.push(...Object.entries(pkgBins).map(([cmd, bin]) => {
                    const resolvedBin = path.join("$PWD", path.relative(this.outdir, subProject.outdir), bin);
                    return [cmd, resolvedBin];
                }));
            }
        });
        const linkTask = this.addTask("workspace:bin:link", {
            steps: bins.map(([cmd, bin]) => ({
                exec: `ln -s ${bin} ${utils_1.NodePackageUtils.command.bin(this.package.packageManager, cmd)} &>/dev/null; exit 0;`,
            })),
        });
        (this.tasks.tryFind("prepare") || this.addTask("prepare")).spawn(linkTask);
    }
    addUpgradeDepsTask() {
        if (this._options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack@^13");
            const upgradeDepsTask = this.addTask(this._options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps", {
                description: "Upgrade dependencies in the monorepo",
            });
            // Run the upgrade task for any non-node subprojects
            const nonNodeSubprojects = this.subprojects.filter((p) => !utils_1.NodePackageUtils.isNodeProject(p));
            if (nonNodeSubprojects.length > 0) {
                this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
                    target: "upgrade",
                    projects: nonNodeSubprojects.map((p) => p.name),
                });
            }
            this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
            // Upgrade node subprojects together
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "npm-check-updates", "--reject projen", "--deep", "--rejectVersion", "0.0.0", "-u", "--dep", "prod,dev,peer,optional,bundle", "--target=minor"));
            // Sync dependency versions across node subprojects
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(`rm ${this.package.lockFile}`);
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.install(this.package.packageManager));
            !this.ejected &&
                upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: this._options.monorepoUpgradeDepsOptions?.syncpackConfig ||
                    syncpack_options_1.Syncpack.DEFAULT_CONFIG,
                readonly: true,
            });
        }
    }
    preSynthesize() {
        this.addUpgradeDepsTask();
        utils_1.NodePackageUtils.removeProjenScript(this);
        this.nxConfigurator.preSynthesize();
        super.preSynthesize();
        if (this._options.workspaceConfig?.linkLocalWorkspaceBins === true) {
            this.linkLocalWorkspaceBins();
        }
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                // Remove any subproject .npmrc files since only the root one matters
                subProject.tryRemoveFile(".npmrc");
                utils_1.NodePackageUtils.removeProjenScript(subProject);
            }
        });
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.installNonNodeDependencies();
        this.resolveLocalBunDependencies();
        // Disable node warnings if configured
        if (this._options.disableNodeWarnings) {
            this.disableNodeWarnings();
        }
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                const subNodeProject = subProject;
                const subNodeProjectResolver = 
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson;
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => {
                    this.subNodeProjectResolves.push(() => subNodeProjectResolver.apply(subNodeProject.package));
                };
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson = () => { };
            }
        });
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        super.postSynthesize();
        this.resolveSubNodeProjects();
    }
    /**
     * Resolve sub `NodePackage` dependencies.
     */
    resolveSubNodeProjects() {
        if (this.subNodeProjectResolves.length) {
            if (!this.package.file.changed) {
                // Force workspace install deps since it would not have been invoked during `postSynthesis`.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
            const completedResolves = this.subNodeProjectResolves.map((resolve) => resolve());
            if (completedResolves.some(Boolean)) {
                // Indicates that a subproject dependency has been resolved from '*', so update the lockfile.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
        }
        this.subNodeProjectResolves = [];
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.yarn?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.yarn?.disableNoHoistBundled !== true &&
            [
                javascript_1.NodePackageManager.YARN,
                javascript_1.NodePackageManager.YARN2,
                javascript_1.NodePackageManager.YARN_BERRY,
                javascript_1.NodePackageManager.YARN_CLASSIC,
            ].includes(this.package.packageManager)) {
            const noHoistBundled = this.subprojects.flatMap((sub) => {
                if (utils_1.ProjectUtils.isNamedInstanceOf(sub, javascript_1.NodeProject)) {
                    return getBundledDeps(sub).flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    installNonNodeDependencies() {
        const installProjects = this.subprojects.filter((project) => !utils_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject) &&
            project.tasks.tryFind("install"));
        if (installProjects.length > 0) {
            // TODO: Install error on clean repo for postinstall (https://cloud.nx.app/runs/MptQr0BxgF) (https://github.com/nrwl/nx/issues/11210)
            const postinstallTask = this.tasks.tryFind("postinstall") ?? this.addTask("postinstall");
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            postinstallTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Suppress Node warnings from being presented in the console when running builds.
     */
    disableNodeWarnings() {
        this.tasks.addEnvironment("NODE_NO_WARNINGS", "1");
        this.subprojects.forEach((subProject) => subProject.tasks.addEnvironment("NODE_NO_WARNINGS", "1"));
    }
    /**
     * Resolve all local workspace dependencies to keep bun happy.
     */
    resolveLocalBunDependencies() {
        if (this.package.packageManager !== javascript_1.NodePackageManager.BUN) {
            return;
        }
        const nodeSubProjectNames = new Set(this.subprojects
            .filter((s) => utils_1.NodePackageUtils.isNodeProject(s))
            .map((s) => s.package.packageName));
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                subProject.deps.all
                    .filter((dep) => nodeSubProjectNames.has(dep.name))
                    .forEach((d) => {
                    switch (d.type) {
                        case projen_1.DependencyType.RUNTIME:
                            subProject.addDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.BUILD:
                            subProject.addDevDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.PEER:
                            subProject.addPeerDeps(`${d.name}@workspace:*`);
                            break;
                        default:
                            console.warn(`Cannot update local dependency due to unsupported type: ${d.type}`);
                    }
                });
            }
        });
    }
}
exports.MonorepoTsProject = MonorepoTsProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoTsProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoTsProject", version: "0.25.8" };
/**
 * Gets bundled dependencies for a given project
 */
function getBundledDeps(project) {
    return project.deps.all.filter((dep) => dep.type === projen_1.DependencyType.BUNDLED);
}
//# sourceMappingURL=data:application/json;base64,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