"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cloneDeep = cloneDeep;
exports.isEmpty = isEmpty;
exports.asUndefinedIfEmpty = asUndefinedIfEmpty;
exports.deepMerge = deepMerge;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const util_1 = require("projen/lib/util");
/**
 * Utility to deeply clone a value
 * @param value Value to clone
 * @returns Cloned value
 */
function cloneDeep(value) {
    return JSON.parse(JSON.stringify(value));
}
/** Indicates if value is considered empty */
function isEmpty(value) {
    if (value == null)
        return true;
    if (typeof value === "boolean") {
        return false;
    }
    if (typeof value === "string") {
        return value.length === 0;
    }
    if (Array.isArray(value)) {
        return value.length === 0;
    }
    if ((0, util_1.isObject)(value)) {
        return Object.keys(value).length === 0;
    }
    return false;
}
/** Replace empty value with undefined */
function asUndefinedIfEmpty(value) {
    if (isEmpty(value))
        return undefined;
    return value;
}
/**
 * Recursively merge objects together into a new object with extends support for appending arrays.
 *
 * This is a clone of [projen/lib/util#deepMerge](https://github.com/projen/projen/blob/55ac3657a270285db63e1a6008b3848b36775626/src/util.ts#L218-L281)
 * with added functionality to support appending arrays.
 *
 * @see [projen/lib/util#deepMerge](https://github.com/projen/projen/blob/55ac3657a270285db63e1a6008b3848b36775626/src/util.ts#L218-L281)
 */
function deepMerge(objects, options) {
    const append = options?.append ?? false;
    const destructive = options?.destructive ?? false;
    objects = objects.map(cloneDeep);
    if (append === false)
        return (0, util_1.deepMerge)(objects, destructive);
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if ((0, util_1.isObject)(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                if (Array.isArray(value)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value);
                    }
                    else {
                        target[key] = value;
                    }
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (Array.isArray(value)) {
                if (Array.isArray(target[key])) {
                    // Append to existing array
                    target[key].push(...value);
                }
                else {
                    // Override with array value
                    target[key] = value;
                }
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
//# sourceMappingURL=data:application/json;base64,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