"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeModulesManifest = exports.readModulesManifest = void 0;
const path_1 = __importDefault(require("path"));
const read_yaml_file_1 = __importDefault(require("read-yaml-file"));
const map_1 = __importDefault(require("ramda/src/map"));
const is_windows_1 = __importDefault(require("is-windows"));
const write_yaml_file_1 = __importDefault(require("write-yaml-file"));
// The dot prefix is needed because otherwise `npm shrinkwrap`
// thinks that it is an extraneous package.
const MODULES_FILENAME = '.modules.yaml';
async function readModulesManifest(modulesDir) {
    const modulesYamlPath = path_1.default.join(modulesDir, MODULES_FILENAME);
    let modules;
    try {
        modules = await (0, read_yaml_file_1.default)(modulesYamlPath);
        if (!modules)
            return modules;
    }
    catch (err) { // eslint-disable-line
        if (err.code !== 'ENOENT') {
            throw err;
        }
        return null;
    }
    if (!modules.virtualStoreDir) {
        modules.virtualStoreDir = path_1.default.join(modulesDir, '.pnpm');
    }
    else if (!path_1.default.isAbsolute(modules.virtualStoreDir)) {
        modules.virtualStoreDir = path_1.default.join(modulesDir, modules.virtualStoreDir);
    }
    switch (modules.shamefullyHoist) {
        case true:
            if (modules.publicHoistPattern == null) {
                modules.publicHoistPattern = ['*'];
            }
            if ((modules.hoistedAliases != null) && !modules.hoistedDependencies) {
                modules.hoistedDependencies = (0, map_1.default)((aliases) => Object.fromEntries(aliases.map((alias) => [alias, 'public'])), modules.hoistedAliases);
            }
            break;
        case false:
            if (modules.publicHoistPattern == null) {
                modules.publicHoistPattern = [];
            }
            if ((modules.hoistedAliases != null) && !modules.hoistedDependencies) {
                modules.hoistedDependencies = {};
                for (const depPath of Object.keys(modules.hoistedAliases)) {
                    modules.hoistedDependencies[depPath] = {};
                    for (const alias of modules.hoistedAliases[depPath]) {
                        modules.hoistedDependencies[depPath][alias] = 'private';
                    }
                }
            }
            break;
    }
    if (!modules.prunedAt) {
        modules.prunedAt = new Date().toUTCString();
    }
    return modules;
}
exports.readModulesManifest = readModulesManifest;
const YAML_OPTS = {
    lineWidth: 1000,
    noCompatMode: true,
    noRefs: true,
    sortKeys: true,
};
async function writeModulesManifest(modulesDir, modules, opts) {
    const modulesYamlPath = path_1.default.join(modulesDir, MODULES_FILENAME);
    const saveModules = { ...modules };
    if (saveModules.skipped)
        saveModules.skipped.sort();
    if (saveModules.hoistPattern == null || saveModules.hoistPattern === '') {
        // Because the YAML writer fails on undefined fields
        delete saveModules.hoistPattern;
    }
    if (saveModules.publicHoistPattern == null) {
        delete saveModules.publicHoistPattern;
    }
    if ((saveModules.hoistedAliases == null) || (saveModules.hoistPattern == null) && (saveModules.publicHoistPattern == null)) {
        delete saveModules.hoistedAliases;
    }
    // We should store the absolute virtual store directory path on Windows
    // because junctions are used on Windows. Junctions will break even if
    // the relative path to the virtual store remains the same after moving
    // a project.
    if (!(0, is_windows_1.default)()) {
        saveModules.virtualStoreDir = path_1.default.relative(modulesDir, saveModules.virtualStoreDir);
    }
    try {
        await (0, write_yaml_file_1.default)(modulesYamlPath, saveModules, {
            ...YAML_OPTS,
            makeDir: opts?.makeModulesDir ?? false,
        });
    }
    catch (err) { // eslint-disable-line
        if (err.code !== 'ENOENT')
            throw err;
    }
}
exports.writeModulesManifest = writeModulesManifest;
//# sourceMappingURL=index.js.map