"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const memory_logger_1 = require("./loggers/memory-logger");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
const DEFAULT_NAG_PACKS = [
    new cdk_nag_1.AwsSolutionsChecks({
        verbose: true,
        reports: true,
    }),
];
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this._extendedNagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
        this.nagPacks = props?.nagPacks ?? DEFAULT_NAG_PACKS;
        aws_cdk_lib_1.Aspects.of(this).add(new PDKNagAspect(this));
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
    addExtendedNagResults(...results) {
        this._extendedNagResults.push(...results);
    }
    /**
     * Returns a list of ExtendedNagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    extendedNagResults() {
        return this._extendedNagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws/pdk.pdk_nag.PDKNagApp", version: "0.25.8" };
class PDKNagAspect {
    constructor(app) {
        this.app = app;
    }
    visit(node) {
        const memoryLogger = new memory_logger_1.MemoryLogger();
        this.app.nagPacks.forEach((nagPack) => {
            // @ts-ignore loggers is private, but since we haven't called "visit" yet it's safe to add another
            nagPack.loggers.push(memoryLogger);
            nagPack.visit(node);
            // @ts-ignore loggers is private, but remove the memory logger to clean up
            nagPack.loggers.pop();
        });
        this.app.addExtendedNagResults(...memoryLogger.results);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        return new PDKNagApp(props);
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws/pdk.pdk_nag.PDKNag", version: "0.25.8" };
//# sourceMappingURL=data:application/json;base64,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