"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Integration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const integration_1 = require("./integration");
const integration_response_sets_1 = require("./integration-response-sets");
const utils_1 = require("../spec/utils");
/**
 * An S3 integration
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/integrating-api-with-aws-services-s3.html
 */
class S3Integration extends integration_1.Integration {
    constructor(props) {
        super();
        this.executionRoleId = "S3IntegrationsExecutionRole";
        this.bucket = props.bucket;
        this.method = props.method;
        this.path = props.path;
        this.integrationResponseSet = props.integrationResponseSet;
    }
    isRole(construct) {
        return "roleArn" in construct && "grantPrincipal" in construct;
    }
    executionRole(scope) {
        // Retrieve or create the shared S3 execution role
        const existingExecutionRole = scope.node.tryFindChild(this.executionRoleId);
        if (existingExecutionRole) {
            if (this.isRole(existingExecutionRole)) {
                return existingExecutionRole;
            }
            throw new Error(`Found construct with ID ${this.executionRoleId} in API scope which was not a role`);
        }
        return new aws_iam_1.Role(scope, this.executionRoleId, {
            assumedBy: new aws_iam_1.ServicePrincipal("apigateway.amazonaws.com"),
        });
    }
    /**
     * Render the S3 integration as a snippet of OpenAPI
     */
    render(props) {
        return {
            type: "AWS",
            httpMethod: (this.method ?? props.method).toUpperCase(),
            uri: (0, utils_1.bucketInvocationUri)(this.bucket, this.path ?? props.path),
            credentials: this.executionRole(props.scope).roleArn,
            requestParameters: {
                // Add every path parameter to the integration request
                ...Object.fromEntries([...props.path.matchAll(/\{([^\}]*)\}/g)]
                    .map((m) => m[1])
                    .map((param) => [
                    `integration.request.path.${param}`,
                    `method.request.path.${param}`,
                ])),
            },
            responses: {
                ...(this.integrationResponseSet ??
                    integration_response_sets_1.IntegrationResponseSets.composite(integration_response_sets_1.IntegrationResponseSets.defaultPassthrough(), integration_response_sets_1.IntegrationResponseSets.s3JsonErrorMessage())).render(props),
            },
        };
    }
    /**
     * Grant API Gateway permissions to invoke the S3 bucket
     */
    grant({ scope, method, path }) {
        const executionRole = this.executionRole(scope);
        // Replace path parameters with * to grant access to arbitrary values for path parameters
        const permissionPath = (this.path ?? path).replace(/{[^\}]*\}/g, "*");
        // Grant read access for GET/HEAD/OPTIONS/TRACE, otherwise write
        if (["get", "head", "options", "trace"].includes((this.method ?? method).toLowerCase())) {
            this.bucket.grantRead(executionRole, permissionPath);
        }
        else {
            this.bucket.grantWrite(executionRole, permissionPath);
        }
    }
}
exports.S3Integration = S3Integration;
_a = JSII_RTTI_SYMBOL_1;
S3Integration[_a] = { fqn: "@aws/pdk.type_safe_api.S3Integration", version: "0.25.8" };
//# sourceMappingURL=data:application/json;base64,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