"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class TypeSafeRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, specPath, operationLookup, defaultAuthorizer, corsOptions, outputSpecBucket, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        const prepareSpecOutputBucket = outputSpecBucket ?? inputSpecAsset.bucket;
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        // Lambda name prefix is truncated to 48 characters (16 below the max of 64)
        const lambdaNamePrefix = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")
            .slice(0, 40)}${this.node.addr.slice(-8).toUpperCase()}`;
        const prepareSpecLambdaName = `${lambdaNamePrefix}PrepSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                prepareSpecOutputBucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "S3 resources have been scoped down to the appropriate prefix in the CDK asset bucket, however * is still needed as since the prepared spec hash is not known until deploy time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:s3:.*/${preparedSpecOutputKeyPrefix}/\*/g`,
                        },
                    ],
                },
            ], true);
        });
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpecHandler", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "./prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${lambdaNamePrefix}PrepSpecProvider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        const serializedCorsOptions = corsOptions && {
            allowHeaders: corsOptions.allowHeaders || [
                ...aws_apigateway_1.Cors.DEFAULT_HEADERS,
                "x-amz-content-sha256",
            ],
            allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
            allowOrigins: corsOptions.allowOrigins,
            statusCode: corsOptions.statusCode || 204,
        };
        const prepareSpecOptions = {
            defaultAuthorizerReference: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, integration]) => [
                operationId,
                {
                    integration: integration.integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                        corsOptions: serializedCorsOptions,
                        operationLookup,
                    }),
                    methodAuthorizer: (0, api_gateway_auth_1.serializeAsAuthorizerReference)(integration.authorizer),
                    options: integration.options,
                },
            ])),
            securitySchemes: (0, api_gateway_auth_1.prepareSecuritySchemes)(this, integrations, defaultAuthorizer, options.apiKeyOptions),
            corsOptions: serializedCorsOptions,
            operationLookup,
            apiKeyOptions: options.apiKeyOptions,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const spec = JSON.parse(fs.readFileSync(specPath, "utf-8"));
        this.extendedApiSpecification = (0, prepare_spec_1.prepareApiSpec)(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: prepareSpecOutputBucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: this.node.tryGetContext("type-safe-api-local")
                ? aws_apigateway_1.ApiDefinition.fromInline(this.extendedApiSpecification)
                : aws_apigateway_1.ApiDefinition.fromBucket(prepareSpecOutputBucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(this, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(this.extendedApiSpecification);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
                operationLookup,
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        (0, api_gateway_integrations_1.getAuthorizerFunctions)(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                        },
                    ],
                },
            ], true);
        });
        ["AwsSolutions-APIG2", "AwsPrototyping-APIGWRequestValidation"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "This construct implements fine grained validation via OpenApi.",
                },
            ], true);
        });
    }
}
exports.TypeSafeRestApi = TypeSafeRestApi;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeRestApi[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeRestApi", version: "0.25.8" };
//# sourceMappingURL=data:application/json;base64,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