"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeApiCommandEnvironment = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const utils_1 = require("./utils");
/**
 * Component for setting up the environemnt for type safe api commands
 */
class TypeSafeApiCommandEnvironment extends projen_1.Component {
    /**
     * Retrieves an instance of TypeSafeApiCommandEnvironment if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, TypeSafeApiCommandEnvironment));
    }
    /**
     * Retrieves an instance of TypeSafeApiCommandEnvironment if one is associated to the given project,
     * otherwise creates a TypeSafeApiCommandEnvironment instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return (TypeSafeApiCommandEnvironment.of(project) ||
            new TypeSafeApiCommandEnvironment(project));
    }
    constructor(project) {
        super(project);
        // Add all environment variables to the task environment
        Object.entries((0, utils_1.getTypeSafeApiTaskEnvironment)()).forEach(([key, value]) => project.tasks.addEnvironment(key, value));
        // Node projects need a dev dependency on PDK to ensure npx resolves to the correct pdk
        if (monorepo_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject)) {
            project.addDevDeps("@aws/pdk@^0");
        }
    }
}
exports.TypeSafeApiCommandEnvironment = TypeSafeApiCommandEnvironment;
//# sourceMappingURL=data:application/json;base64,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