"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCodegenCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../../../monorepo");
const runtime_version_utils_1 = require("../runtime-version-utils");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    OtherGenerators["TYPESCRIPT_ASYNC_CDK_INFRASTRUCTURE"] = "typescript-async-cdk-infrastructure";
    OtherGenerators["PYTHON_ASYNC_CDK_INFRASTRUCTURE"] = "python-async-cdk-infrastructure";
    OtherGenerators["JAVA_ASYNC_CDK_INFRASTRUCTURE"] = "java-async-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
    OtherGenerators["TYPESCRIPT_ASYNC_LAMBDA_HANDLERS"] = "typescript-async-lambda-handlers";
    OtherGenerators["PYTHON_ASYNC_LAMBDA_HANDLERS"] = "python-async-lambda-handlers";
    OtherGenerators["JAVA_ASYNC_LAMBDA_HANDLERS"] = "java-async-lambda-handlers";
    // Async runtime
    OtherGenerators["TYPESCRIPT_ASYNC_RUNTIME"] = "typescript-async-runtime";
    OtherGenerators["PYTHON_ASYNC_RUNTIME"] = "python-async-runtime";
    OtherGenerators["JAVA_ASYNC_RUNTIME"] = "java-async-runtime";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
/**
 * Built in scripts.
 * If adding a script here, ensure you map it in TypeSafeApiProject (in /projenrc/projects)
 */
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api generate-mock-data";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api copy-gradle-wrapper";
    TypeSafeApiScript["COPY_ASYNC_SMITHY_TRANSFORMER"] = "type-safe-api copy-async-smithy-transformer";
    TypeSafeApiScript["GENERATE_ASYNCAPI_SPEC"] = "type-safe-api generate-asyncapi-spec";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: monorepo_1.ProjectUtils.getPdkVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const buildCodegenCommandArgs = (options) => {
    const metadata = options.metadata
        ? ` --metadata '${JSON.stringify(options.metadata)}'`
        : "";
    return `--specPath ${options.specPath} --outputPath . --templateDirs ${options.templateDirs
        .map((t) => `"${t}"`)
        .join(" ")}${metadata}`;
};
exports.buildCodegenCommandArgs = buildCodegenCommandArgs;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --maxArrayLength ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--specPath ${options.specPath} --outputPath ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
    "x-handlers-node-lambda-runtime-version": typescript
        ? runtime_version_utils_1.RuntimeVersionUtils.NODE.getLambdaRuntime(typescript.runtimeVersion)
        : "",
    "x-handlers-python-lambda-runtime-version": python
        ? runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getLambdaRuntime(python.runtimeVersion)
        : "",
    "x-handlers-java-lambda-runtime-version": java
        ? runtime_version_utils_1.RuntimeVersionUtils.JAVA.getLambdaRuntime(java.runtimeVersion)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,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