"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const types_1 = require("../../types");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedPythonHandlersBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            pytest: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
            ...options,
            sample: false,
            poetry: true,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.PythonVersion.PYTHON_3_11;
        this.tstDir = "test";
        if (options.pytest ?? true) {
            // Pytest fails with exit code 5 when there are no tests.
            // We want to allow users to delete all their tests, or to upgrade an existing project without breaking their build
            // See: https://github.com/pytest-dev/pytest/issues/2393
            this.testTask.reset(`pytest ${this.tstDir}/ || ([ $? = 5 ] && exit 0 || exit $?)`);
        }
        [
            runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPythonDependencyVersion(this.runtimeVersion),
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // Write __init__.py as sample code
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        // Package task to build a distributable which can be deployed to lambda
        this.packageTask.exec("mkdir -p dist/lambda && rm -rf dist/lambda/*");
        this.packageTask.exec(`cp -r ${this.moduleName} dist/lambda/${this.moduleName}`);
        this.packageTask.exec("poetry export --without-hashes --format=requirements.txt | sed -E 's/^-e[[:space:]]+//' > dist/lambda/requirements.txt");
        // Select the platform based on the specified architecture, defaulting to x86_64
        // See: https://docs.aws.amazon.com/lambda/latest/dg/python-package.html#python-package-native-libraries
        const platform = options.architecture === types_1.Architecture.ARM_64
            ? "manylinux2014_aarch64"
            : "manylinux2014_x86_64";
        this.packageTask.exec(`pip install -r dist/lambda/requirements.txt --target dist/lambda --upgrade --platform ${platform} --only-binary :all: --python-version ${runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPipPackagingPythonVersion(this.runtimeVersion)}`);
    }
}
exports.GeneratedPythonHandlersBaseProject = GeneratedPythonHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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