"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonCdkInfrastructureBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedPythonCdkInfrastructureBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        [
            "aws_pdk@^0",
            "constructs@^10",
            "aws-cdk-lib@^2",
            "cdk-nag@^2",
            "python@^3.9",
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        const generateTask = this.addTask("generate");
        this.generateTask = generateTask;
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        if (!options.commitGeneratedCode) {
            // Ignore the generated code
            this.gitignore.addPatterns(this.moduleName, "mocks");
        }
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py and README.md as our first install step to keep poetry happy until the
        // generator overwrites it.
        ["install", "install:ci"].forEach((t) => this.tasks
            .tryFind(t)
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py README.md`));
    }
}
exports.GeneratedPythonCdkInfrastructureBaseProject = GeneratedPythonCdkInfrastructureBaseProject;
//# sourceMappingURL=data:application/json;base64,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