"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const python_1 = require("projen/lib/python");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Python project containing types generated using OpenAPI Generator CLI
 */
class GeneratedPythonRuntimeBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Add dependencies required by the client
        [
            "python-dateutil@~2.8.2",
            "pydantic@^2.5.2",
            "aenum@^3.1.11",
            "urllib3@~1.26.7",
            `aws-lambda-powertools@{extras=["tracer", "aws-sdk"],version="^2.28.0"}`,
            "python@^3.9",
        ].forEach((dep) => this.addDependency(dep));
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        if (!this.options.commitGeneratedCode) {
            // Ignore all the generated code
            this.gitignore.addPatterns(this.moduleName, "docs", "README.md");
        }
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py and README.md as our first install step to keep poetry happy until the
        // generator overwrites it.
        ["install", "install:ci"].forEach((t) => this.tasks
            .tryFind(t)
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py README.md`));
    }
}
exports.GeneratedPythonRuntimeBaseProject = GeneratedPythonRuntimeBaseProject;
//# sourceMappingURL=data:application/json;base64,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