import numpy as np
import pandas as pd
import pytest

from gwtransport.advection import (
    extraction_to_infiltration,
    extraction_to_infiltration_series,
    gamma_extraction_to_infiltration,
    gamma_infiltration_to_extraction,
    infiltration_to_extraction,
    infiltration_to_extraction_series,
)
from gwtransport.utils import compute_time_edges

# ===============================================================================
# FIXTURES
# ===============================================================================


@pytest.fixture
def sample_time_series():
    """Create sample time series data for testing."""
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    concentration = pd.Series(np.sin(np.linspace(0, 4 * np.pi, len(dates))) + 2, index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)  # Constant flow of 100 m3/day
    return concentration, flow


@pytest.fixture
def gamma_params():
    """Sample gamma distribution parameters."""
    return {
        "alpha": 10.0,  # Shape parameter (smaller for reasonable mean)
        "beta": 10.0,  # Scale parameter (gives mean = alpha * beta = 100)
        "n_bins": 10,  # Number of bins
    }


# ===============================================================================
# INFILTRATION_TO_EXTRACTION_SERIES FUNCTION TESTS
# ===============================================================================


def test_infiltration_to_extraction_series_output_structure():
    """Test that infiltration_to_extraction_series returns correct DatetimeIndex structure."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    assert isinstance(tedges_out, pd.DatetimeIndex)
    assert len(tedges_out) == len(tedges)
    assert isinstance(tedges_out[0], pd.Timestamp)


def test_infiltration_to_extraction_series_constant_input():
    """Test constant concentration produces constant output with proper time shift."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    # Time shift should be residence time = pore_volume / flow = 500 / 100 = 5 days
    expected_shift = pd.Timedelta(days=5)
    actual_shift = tedges_out[0] - tedges[0]
    assert actual_shift == expected_shift


def test_infiltration_to_extraction_series_retardation_factor():
    """Test retardation factor doubles residence time."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out_no_retard = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
        retardation_factor=1.0,
    )

    tedges_out_retard = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
        retardation_factor=2.0,
    )

    # Retardation factor of 2 should double time shift
    shift_no_retard = tedges_out_no_retard[0] - tedges[0]
    shift_retard = tedges_out_retard[0] - tedges[0]
    assert shift_retard == shift_no_retard * 2


def test_infiltration_to_extraction_series_pandas_series_input():
    """Test function accepts pandas Series as input."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = pd.Series(np.ones(len(dates)) * 100.0, index=dates)

    tedges_out = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    assert isinstance(tedges_out, pd.DatetimeIndex)
    assert len(tedges_out) == len(tedges)


def test_infiltration_to_extraction_series_time_edges_consistency():
    """Test output time edges are monotonically increasing (excluding NaT values)."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    # Output time edges should be monotonically increasing (excluding NaT values)
    time_diffs = tedges_out.to_series().diff()[1:]
    valid_diffs = time_diffs.dropna()
    assert (valid_diffs > pd.Timedelta(0)).all(), "Valid output time edges must be monotonically increasing"


# ===============================================================================
# EXTRACTION_TO_INFILTRATION_SERIES FUNCTION TESTS
# ===============================================================================


def test_extraction_to_infiltration_series_output_structure():
    """Test that extraction_to_infiltration_series returns correct DatetimeIndex structure."""
    # Use dates starting later to ensure we have valid backward data
    dates = pd.date_range(start="2020-01-10", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    assert isinstance(tedges_out, pd.DatetimeIndex)
    assert len(tedges_out) == len(tedges)
    # Check that we have at least some valid timestamps (not all NaT)
    assert tedges_out.notna().any()


def test_extraction_to_infiltration_series_constant_input():
    """Test constant concentration produces constant output with proper time shift backward."""
    dates = pd.date_range(start="2020-01-10", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    # Time shift should be residence time = pore_volume / flow = 500 / 100 = 5 days (backward)
    # Check using the last valid index instead of first (which may be NaT)
    expected_shift = pd.Timedelta(days=5)
    actual_shift = tedges[-1] - tedges_out[-1]
    assert actual_shift == expected_shift


def test_extraction_to_infiltration_series_retardation_factor():
    """Test retardation factor doubles residence time (backward shift)."""
    dates = pd.date_range(start="2020-01-15", end="2020-01-25", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out_no_retard = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
        retardation_factor=1.0,
    )

    tedges_out_retard = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
        retardation_factor=2.0,
    )

    # Retardation factor of 2 should double time shift (backward)
    # Check using the last valid index instead of first (which may be NaT)
    shift_no_retard = tedges[-1] - tedges_out_no_retard[-1]
    shift_retard = tedges[-1] - tedges_out_retard[-1]
    assert shift_retard == shift_no_retard * 2


def test_extraction_to_infiltration_series_pandas_series_input():
    """Test function accepts pandas Series as input."""
    # Use dates starting later to have enough backward data
    dates = pd.date_range(start="2020-01-10", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = pd.Series(np.ones(len(dates)) * 100.0, index=dates)

    tedges_out = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    assert isinstance(tedges_out, pd.DatetimeIndex)
    assert len(tedges_out) == len(tedges)


def test_extraction_to_infiltration_series_time_edges_consistency():
    """Test output time edges are monotonically increasing (excluding NaT values)."""
    # Use dates starting later to have sufficient backward data
    dates = pd.date_range(start="2020-01-10", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    flow = np.ones(len(dates)) * 100.0

    tedges_out = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=500.0,
    )

    # Output time edges should be monotonically increasing (excluding NaT values)
    time_diffs = tedges_out.to_series().diff()[1:]
    valid_diffs = time_diffs.dropna()
    assert (valid_diffs > pd.Timedelta(0)).all(), "Valid output time edges must be monotonically increasing"


def test_extraction_to_infiltration_series_symmetry_with_infiltration():
    """Test symmetry: roundtrip time shift recovers original time edges in valid region."""
    # Test that forward shift (infiltration → extraction) followed by
    # backward shift (extraction → infiltration) recovers original time edges
    # in the region where both operations have sufficient data (analytical solution)
    dates = pd.date_range(start="2020-01-01", end="2020-01-30", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    flow = np.ones(len(dates)) * 100.0
    pore_volume = 500.0  # 5 days residence time
    retardation_factor = 1.5  # 7.5 days effective residence time

    # Forward: infiltration → extraction (shift forward in time)
    tedges_extraction = infiltration_to_extraction_series(
        flow=flow,
        tedges=tedges,
        aquifer_pore_volume=pore_volume,
        retardation_factor=retardation_factor,
    )

    # Backward: extraction → infiltration (shift backward in time)
    tedges_recovered = extraction_to_infiltration_series(
        flow=flow,
        tedges=tedges_extraction,
        aquifer_pore_volume=pore_volume,
        retardation_factor=retardation_factor,
    )

    # Find valid (non-NaT) region in recovered time edges
    valid_mask = tedges_recovered.notna()

    # In the valid region, roundtrip must recover original exactly (analytical solution)
    # Verify at least some central portion is recovered (residence time is ~7.5 days,
    # so we lose edges but keep center)
    valid_count = valid_mask.sum()
    assert valid_count >= 15, f"Expected at least 15 valid recovered time edges, got {valid_count}"

    # Compare only valid elements - these must match exactly
    pd.testing.assert_index_equal(
        tedges_recovered[valid_mask],
        tedges[valid_mask],
        check_exact=True,
        obj="Roundtrip time edges must exactly match original in valid region",
    )


def test_gamma_infiltration_to_extraction_basic_functionality():
    """Test basic functionality of gamma_infiltration_to_extraction."""
    # Create shorter test data with aligned cin and flow
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Create cout_tedges with different alignment to avoid edge effects
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-15", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,  # Shape parameter
        beta=10.0,  # Scale parameter (mean = 100)
        n_bins=5,
    )

    # Check output type and length
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)

    # Check output values are non-negative (ignoring NaN values)
    valid_values = cout[~np.isnan(cout)]
    assert np.all(valid_values >= 0)


def test_gamma_infiltration_to_extraction_with_mean_std():
    """Test gamma_infiltration_to_extraction using mean and std parameters."""
    # Create test data
    dates = pd.date_range(start="2020-01-01", end="2020-01-30", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-10", end="2020-01-25", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    mean = 100.0  # Smaller mean for reasonable residence time
    std = 20.0

    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        mean=mean,
        std=std,
        n_bins=5,
    )

    # Check output type and length
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)


def test_gamma_infiltration_to_extraction_retardation_factor():
    """Test gamma_infiltration_to_extraction with different retardation factors."""
    # Create test data - use year-long series for robust results
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-03-01", end="2020-10-31", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Use a step function to see retardation effects
    cin_values = np.ones(len(dates))
    cin_values[90:] = 2.0  # Step change on day 91 (April 1)
    cin = pd.Series(cin_values, index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Compare results with different retardation factors
    cout1 = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        retardation_factor=1.0,
        n_bins=20,
    )

    cout2 = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        retardation_factor=2.0,
        n_bins=20,
    )

    # Explicit validation
    valid_mask = ~np.isnan(cout1) & ~np.isnan(cout2)
    valid_count = np.sum(valid_mask)
    assert valid_count >= 200, f"Expected at least 200 valid overlap bins, got {valid_count}"

    # Extract valid values
    cout1_valid = cout1[valid_mask]
    cout2_valid = cout2[valid_mask]

    # Test that step timing is different (max absolute difference)
    max_diff = np.max(np.abs(cout1_valid - cout2_valid))
    assert max_diff > 0.1, f"Expected max difference > 0.1 due to shifted step, got {max_diff:.3f}"

    # Test that variation is present (step was detected)
    std1 = np.std(cout1_valid)
    std2 = np.std(cout2_valid)
    assert std1 > 0.1, f"Expected std1 > 0.1 showing step, got {std1:.3f}"
    assert std2 > 0.1, f"Expected std2 > 0.1 showing step, got {std2:.3f}"


def test_gamma_infiltration_to_extraction_constant_input():
    """Test gamma_infiltration_to_extraction with constant input concentration."""
    # Create test data with longer input period
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period starts later to allow for residence time
    cout_dates = pd.date_range(start="2020-06-01", end="2020-11-30", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        n_bins=20,
    )

    # Explicit validation
    assert len(cout) == len(cout_dates), f"Expected {len(cout_dates)} output bins, got {len(cout)}"
    valid_count = np.sum(~np.isnan(cout))
    assert valid_count >= 150, f"Expected at least 150 valid bins for 6-month extraction, got {valid_count}"

    # Output should also be constant where valid (constant input preserved)
    valid_values = cout[~np.isnan(cout)]
    mean_cout = np.mean(valid_values)
    assert abs(mean_cout - 1.0) < 0.1, f"Expected mean ~1.0 (preserved from constant input), got {mean_cout:.3f}"

    # Low variation expected for constant input
    std_cout = np.std(valid_values)
    assert std_cout < 0.05, f"Expected std < 0.05 for constant input, got {std_cout:.3f}"


def test_gamma_infiltration_to_extraction_missing_parameters():
    """Test that gamma_infiltration_to_extraction raises appropriate errors for missing parameters."""
    # Create test data
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-08", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Test missing both alpha/beta and mean/std
    with pytest.raises(ValueError):
        gamma_infiltration_to_extraction(cin=cin, tedges=tedges, cout_tedges=cout_tedges, flow=flow)


# ===============================================================================
# GAMMA_INFILTRATION_TO_EXTRACTION FUNCTION - ANALYTICAL SOLUTION TESTS
# ===============================================================================


def test_gamma_infiltration_to_extraction_analytical_mean_residence_time():
    """Test gamma_infiltration_to_extraction with analytical mean residence time."""
    # Create constant input
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period starts later to capture steady state
    cout_dates = pd.date_range(start="2020-06-01", end="2020-11-30", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Constant input and flow
    cin = pd.Series([10.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Gamma distribution parameters
    # Mean residence time = alpha * beta / flow = 10 * 10 / 100 = 1 day
    alpha = 10.0
    beta = 10.0

    # Run gamma_infiltration_to_extraction
    cout = gamma_infiltration_to_extraction(
        cin=cin,
        flow=flow,
        tedges=tedges,
        cout_tedges=cout_tedges,
        alpha=alpha,
        beta=beta,
        n_bins=20,
        retardation_factor=1.0,
    )

    # Explicit validation
    valid_mask = ~np.isnan(cout)
    valid_count = np.sum(valid_mask)
    assert valid_count >= 150, f"Expected at least 150 valid bins for 6-month extraction, got {valid_count}"

    # Extract stable region (last 30 valid points for steady state)
    valid_indices = np.where(valid_mask)[0]
    assert len(valid_indices) >= 30, f"Need at least 30 valid points, got {len(valid_indices)}"

    stable_indices = valid_indices[-30:]
    stable_region = cout[stable_indices]

    # Analytical solution: for constant input, output should eventually equal input
    mean_output = np.mean(stable_region)
    assert abs(mean_output - 10.0) < 0.5, f"Expected mean ~10.0 in steady state, got {mean_output:.2f}"

    # Variance should be small in steady state
    std_output = np.std(stable_region)
    assert std_output < 0.5, f"Expected std < 0.5 in steady state, got {std_output:.2f}"


# ===============================================================================
# DISTRIBUTION_INFILTRATION_TO_EXTRACTION FUNCTION TESTS
# ===============================================================================


def test_infiltration_to_extraction_basic_functionality():
    """Test basic functionality of infiltration_to_extraction."""
    # Create test data with aligned cin and flow
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Create cout_tedges with different alignment
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-09", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    # Use smaller pore volumes for reasonable residence times (1-3 days)
    aquifer_pore_volumes = np.array([100.0, 200.0, 300.0])

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Check output type and length
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)


def test_infiltration_to_extraction_constant_input():
    """Test infiltration_to_extraction with constant input concentration."""
    # Create longer time series for better testing
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Create cout_tedges starting later
    cout_dates = pd.date_range(start="2020-06-01", end="2020-11-30", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)) * 5.0, index=dates)  # Constant concentration
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)  # Constant flow
    aquifer_pore_volumes = np.array([500.0, 1000.0])  # Two pore volumes

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Explicit validation
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates), f"Expected {len(cout_dates)} output bins, got {len(cout)}"

    valid_count = np.sum(~np.isnan(cout))
    assert valid_count >= 150, f"Expected at least 150 valid bins for 6-month extraction, got {valid_count}"

    # Test constant input preservation
    valid_outputs = cout[~np.isnan(cout)]
    mean_cout = np.mean(valid_outputs)
    assert abs(mean_cout - 5.0) < 0.5, f"Expected mean ~5.0 (preserved from constant input), got {mean_cout:.3f}"

    # Test low variation for constant input
    std_cout = np.std(valid_outputs)
    assert std_cout < 0.5, f"Expected std < 0.5 for constant input, got {std_cout:.3f}"


def test_infiltration_to_extraction_single_pore_volume():
    """Test infiltration_to_extraction with single pore volume."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-10", end="2020-01-15", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.sin(np.linspace(0, 2 * np.pi, len(dates))) + 2, index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    # Use smaller pore volume for reasonable residence time (5 days)
    aquifer_pore_volumes = np.array([500.0])  # Single pore volume

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)


def test_infiltration_to_extraction_retardation_factor():
    """Test infiltration_to_extraction with different retardation factors."""
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-06-01", end="2020-11-30", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    aquifer_pore_volumes = np.array([1000.0, 2000.0])

    # Test different retardation factors
    cout1 = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    cout2 = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=2.0,
    )

    # Results should be different for different retardation factors
    assert isinstance(cout1, np.ndarray)
    assert isinstance(cout2, np.ndarray)
    assert len(cout1) == len(cout2)


def test_infiltration_to_extraction_error_conditions():
    """Test infiltration_to_extraction error conditions."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    aquifer_pore_volumes = np.array([1000.0])

    # Test mismatched tedges length
    wrong_tedges = tedges[:-2]  # Too short
    with pytest.raises(ValueError, match="tedges must have one more element than cin"):
        infiltration_to_extraction(
            cin=cin.values,
            flow=flow.values,
            tedges=wrong_tedges,
            cout_tedges=cout_tedges,
            aquifer_pore_volumes=aquifer_pore_volumes,
        )

    # Test mismatched flow and tedges
    wrong_flow = flow[:-2]  # Too short
    with pytest.raises(ValueError, match="tedges must have one more element than flow"):
        infiltration_to_extraction(
            cin=cin.values,
            flow=wrong_flow.values,
            tedges=tedges,
            cout_tedges=cout_tedges,
            aquifer_pore_volumes=aquifer_pore_volumes,
        )


# ===============================================================================
# DISTRIBUTION_FORWARD FUNCTION - EDGE CASE TESTS
# ===============================================================================


def test_infiltration_to_extraction_no_temporal_overlap():
    """Test infiltration_to_extraction returns NaN when no temporal overlap exists."""
    # Create cin/flow in early 2020
    early_dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=early_dates, number_of_bins=len(early_dates))

    # Create cout_tedges much later (no possible overlap)
    late_dates = pd.date_range(start="2020-12-01", end="2020-12-05", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=late_dates, number_of_bins=len(late_dates))

    cin = pd.Series(np.ones(len(early_dates)), index=early_dates)
    flow = pd.Series(np.ones(len(early_dates)) * 100, index=early_dates)
    aquifer_pore_volumes = np.array([100.0])  # Small pore volume

    # No temporal overlap should return all NaN values
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array of NaN values
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(late_dates)
    assert np.all(np.isnan(cout))


def test_infiltration_to_extraction_zero_concentrations():
    """Test infiltration_to_extraction preserves zero concentrations with aligned residence time."""
    # Create longer time series for realistic residence times
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # cout_tedges later to allow residence time effects
    cout_dates = pd.date_range(start="2020-01-10", end="2020-12-20", freq="D")  # Overlap with input period
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Create cin with repeating pattern: [1.0, 0.0, 2.0] (3-day period)
    cin_pattern = np.array([1.0, 0.0, 2.0])
    cin_values = np.tile(cin_pattern, len(dates) // len(cin_pattern) + 1)[: len(dates)]
    cin = pd.Series(cin_values, index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Use pore volume that gives residence time matching pattern period (3 days)
    # This ensures zeros align and are preserved in output
    aquifer_pore_volumes = np.array([300.0])  # 300/100 = 3 day residence time

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Explicit validation
    valid_count = np.sum(~np.isnan(cout))
    assert valid_count == len(cout_dates), f"Expected {len(cout_dates)} valid bins, got {valid_count}"

    # Check that zero concentrations are preserved (not converted to NaN)
    valid_results = cout[~np.isnan(cout)]
    has_zeros = np.sum(valid_results == 0.0)
    assert has_zeros >= 80, f"Expected at least 80 zero values preserved from input pattern, got {has_zeros}"

    # Check range of concentration values matches input pattern
    assert np.all(valid_results >= 0.0), "All concentrations should be non-negative"
    assert np.all(valid_results <= 2.0), "All concentrations should be within input range [0, 2]"

    # Verify output pattern matches input pattern (exact preservation with aligned residence time)
    unique_values = np.unique(valid_results)
    expected_values = np.array([0.0, 1.0, 2.0])
    np.testing.assert_allclose(
        unique_values, expected_values, rtol=1e-10, err_msg="Output should preserve exact input pattern values"
    )


def test_infiltration_to_extraction_extreme_conditions():
    """Test infiltration_to_extraction handles extreme conditions gracefully."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-05", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    cout_tedges = tedges.copy()

    cin = pd.Series([1.0, 2.0, 3.0, 4.0, 5.0], index=dates)
    flow = pd.Series([1000.0, 0.1, 1000.0, 0.1, 1000.0], index=dates)
    aquifer_pore_volumes = np.array([10.0, 100000.0, 50.0])

    # Should handle extreme conditions gracefully
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return valid array (may contain NaN values)
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(dates)


def test_infiltration_to_extraction_extreme_pore_volumes():
    """Test infiltration_to_extraction handles extremely large pore volumes gracefully."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-07", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Extremely large pore volumes that create invalid infiltration edges
    aquifer_pore_volumes = np.array([1e10, 1e12, 1e15])

    # Should handle extreme pore volumes gracefully
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array (likely all NaN due to extreme residence times)
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)
    # With extremely large pore volumes, all outputs should be NaN
    assert np.all(np.isnan(cout))


def test_infiltration_to_extraction_zero_flow():
    """Test infiltration_to_extraction handles zero flow values gracefully."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-07", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.zeros(len(dates)), index=dates)  # Zero flow
    aquifer_pore_volumes = np.array([1000.0])

    # Zero flow creates infinite residence times but should be handled gracefully
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array (likely all NaN due to infinite residence times)
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)
    # With zero flow, all outputs should be NaN
    assert np.all(np.isnan(cout))


def test_infiltration_to_extraction_mixed_pore_volumes():
    """Test infiltration_to_extraction handles mixed pore volumes with varying overlaps."""
    # Longer time series for cin/flow
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Short cout period - only some pore volumes will have overlap
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-10", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Mix of small and large pore volumes - large ones create minimal overlap
    aquifer_pore_volumes = np.array([10.0, 100.0, 50000.0, 100000.0])

    # Should handle mixed pore volumes gracefully
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Explicit validation
    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates), f"Expected {len(cout_dates)} output bins, got {len(cout)}"

    # Some values should be valid (from small pore volumes with quick transport)
    valid_count = np.sum(~np.isnan(cout))
    assert valid_count >= 3, f"Expected at least 3 valid bins from small pore volumes, got {valid_count}"

    # Check all valid values are non-negative
    valid_values = cout[~np.isnan(cout)]
    assert np.all(valid_values >= 0), "All valid concentrations should be non-negative"


# ===============================================================================
# DISTRIBUTION_FORWARD FUNCTION - ANALYTICAL SOLUTION TESTS
# ===============================================================================


def test_infiltration_to_extraction_analytical_mass_conservation():
    """Test infiltration_to_extraction mass conservation with pulse input."""
    # Create pulse input (finite mass)
    dates = pd.date_range(start="2020-01-01", end="2020-01-30", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Long output period to capture entire pulse
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-25", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Pulse input: concentration for 5 days, then zero
    cin_values = np.zeros(len(dates))
    cin_values[5:10] = 8.0  # Pulse from day 6-10
    cin = pd.Series(cin_values, index=dates)

    # Constant flow
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Multiple pore volumes
    aquifer_pore_volumes = np.array([100.0, 200.0, 300.0])  # 1, 2, 3 day residence times

    # Run infiltration_to_extraction
    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Mass conservation check
    # Input mass = concentration * flow * time (for each time step)
    dt = 1.0  # 1 day time steps
    input_mass = np.sum(cin_values * flow.values * dt)

    # Output mass = concentration * flow * time (for each time step)
    # Use average flow for output period
    output_flow = np.mean(flow.values)
    valid_mask = ~np.isnan(cout)
    output_mass = np.sum(cout[valid_mask] * output_flow * dt)

    # Check mass conservation (within 20% due to discretization and edge effects)
    if input_mass > 0:
        mass_error = abs(output_mass - input_mass) / input_mass
        assert mass_error < 0.3, f"Mass conservation error {mass_error:.2f} > 0.3"


def test_infiltration_to_extraction_known_constant_delay():
    """Test infiltration_to_extraction with known constant delay scenario."""
    # Create a simple scenario where we know the exact outcome
    # 10 days of data, constant flow, single pore volume
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period starts after the delay
    cout_dates = pd.date_range(start="2020-01-06", end="2020-01-10", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Constant flow and known pore volume that gives exactly 1 day residence time
    flow_rate = 100.0  # m3/day
    pore_volume = 100.0  # m3 -> residence time = 100/100 = 1 day

    # Step function: concentration jumps from 1 to 5 on day 5
    cin_values = [1.0, 1.0, 1.0, 1.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0]
    cin = pd.Series(cin_values, index=dates)
    flow = pd.Series([flow_rate] * len(dates), index=dates)
    aquifer_pore_volumes = np.array([pore_volume])

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # With 1-day residence time, the step change on day 5 should appear on day 6
    # Output days 6-10 correspond to infiltration days 5-9
    # So we expect outputs close to 5.0 (after the step change)
    valid_count = np.sum(~np.isnan(cout))
    assert valid_count >= 4, f"Expected at least 4 valid bins for 1-day delay, got {valid_count}"

    valid_outputs = cout[~np.isnan(cout)]
    mean_output = np.mean(valid_outputs)
    assert abs(mean_output - 5.0) < 0.5, f"Expected mean ~5.0 after step, got {mean_output:.3f}"


def test_infiltration_to_extraction_known_average_of_pore_volumes():
    """Test infiltration_to_extraction averages multiple pore volumes correctly."""
    # Simple scenario where we can predict the averaging behavior
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period in the middle to ensure overlap
    cout_dates = pd.date_range(start="2020-01-10", end="2020-01-15", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Constant concentration and flow
    cin = pd.Series([10.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Two identical pore volumes - average should equal the single pore volume result
    single_pv = np.array([500.0])
    double_pv = np.array([500.0, 500.0])

    cout_single = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=single_pv,
        retardation_factor=1.0,
    )

    cout_double = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=double_pv,
        retardation_factor=1.0,
    )

    # Results should be nearly identical (averaging two identical contributions)
    valid_mask = ~np.isnan(cout_single) & ~np.isnan(cout_double)
    if np.any(valid_mask):
        np.testing.assert_allclose(
            cout_single[valid_mask],
            cout_double[valid_mask],
            rtol=1e-10,
            err_msg="Averaging identical pore volumes should give same result as single pore volume",
        )


def test_infiltration_to_extraction_known_zero_input_gives_zero_output():
    """Test infiltration_to_extraction with zero input gives zero output."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-08", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Zero concentration everywhere
    cin = pd.Series([0.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)
    aquifer_pore_volumes = np.array([200.0, 400.0])

    cout = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Explicit validation - zero input should give zero output
    valid_count = np.sum(~np.isnan(cout))
    assert valid_count == len(cout_dates), f"Expected {len(cout_dates)} valid bins, got {valid_count}"

    valid_outputs = cout[~np.isnan(cout)]
    np.testing.assert_allclose(valid_outputs, 0.0, atol=1e-15, err_msg="Zero input should produce zero output")


def test_infiltration_to_extraction_known_retardation_effect():
    """Test infiltration_to_extraction retardation factor effect."""
    # Create longer time series to capture retardation effects
    dates = pd.date_range(start="2020-01-01", end="2020-01-30", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period covers a wide range to catch both retarded and non-retarded responses
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-25", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Step function: concentration jumps from 0 to 10 on day 10
    cin_values = [0.0] * len(dates)
    for i in range(9, len(dates)):  # Days 10 onwards (index 9+)
        cin_values[i] = 10.0
    cin = pd.Series(cin_values, index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Pore volume that gives reasonable residence time
    pore_volume = 200.0  # residence time = 200/100 = 2 days
    aquifer_pore_volumes = np.array([pore_volume])

    # Test different retardation factors
    cout_no_retard = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    cout_retarded = infiltration_to_extraction(
        cin=cin.values,
        flow=flow.values,
        tedges=tedges,
        cout_tedges=cout_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=2.0,
    )

    # Basic test - both should return valid arrays
    assert isinstance(cout_no_retard, np.ndarray)
    assert isinstance(cout_retarded, np.ndarray)
    assert len(cout_no_retard) == len(cout_dates)
    assert len(cout_retarded) == len(cout_dates)


# ===============================================================================
# COMPARISON TESTS BETWEEN FORWARD AND DISTRIBUTION_FORWARD
# ===============================================================================


def test_time_edge_consistency():
    """Test that time edges are handled consistently."""
    # Create test data with proper temporal alignment
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period starts later to avoid edge effects
    cout_dates = pd.date_range(start="2020-01-05", end="2020-01-15", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Test with consistent time edges
    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        n_bins=5,
    )

    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(cout_dates)


def test_conservation_properties():
    """Test mass conservation properties where applicable."""
    # Create test data with longer time series for better conservation
    dates = pd.date_range(start="2020-01-01", end="2021-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Output period covers most of the second year to capture steady state
    cout_dates = pd.date_range(start="2021-01-01", end="2021-11-30", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    cin = pd.Series(np.ones(len(dates)), index=dates)  # Constant input
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)  # Constant flow

    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        n_bins=10,
    )

    # For constant input and flow, output should eventually stabilize
    # Check the latter part of the series where it should be stable
    valid_mask = ~np.isnan(cout)
    if np.sum(valid_mask) > 100:  # If we have enough valid values
        stable_region = cout[valid_mask][-100:]  # Last 100 valid values
        assert np.std(stable_region) < 0.1  # Should be relatively stable


def test_empty_series():
    """Test handling of empty series."""
    empty_cin = pd.Series([], dtype=float)

    # This should handle gracefully or raise appropriate error
    with pytest.raises((ValueError, IndexError)):
        # Create tedges - this should fail for empty series
        compute_time_edges(
            tedges=None, tstart=None, tend=pd.DatetimeIndex(empty_cin.index), number_of_bins=len(empty_cin)
        )


def test_mismatched_series_lengths():
    """Test handling of mismatched series lengths."""
    # Create input data with longer period
    dates_cin = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=dates_cin, number_of_bins=len(dates_cin))

    # Create output data with shorter, offset period
    dates_cout = pd.date_range(start="2020-01-05", end="2020-01-10", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=dates_cout, number_of_bins=len(dates_cout))

    cin = pd.Series(np.ones(len(dates_cin)), index=dates_cin)
    flow = pd.Series(np.ones(len(dates_cin)) * 100, index=dates_cin)

    # This should work - the function should handle different output lengths
    cout = gamma_infiltration_to_extraction(
        cin=cin,
        tedges=cin_tedges,
        cout_tedges=cout_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
    )

    assert isinstance(cout, np.ndarray)
    assert len(cout) == len(dates_cout)


# ===============================================================================
# DISTRIBUTION_BACKWARD FUNCTION TESTS (MIRROR OF DISTRIBUTION_FORWARD)
# ===============================================================================


def test_extraction_to_infiltration_basic_functionality():
    """Test basic functionality of extraction_to_infiltration."""
    # Create test data with aligned cout and flow
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Create cin_tedges with different alignment
    cint_dates = pd.date_range(start="2019-12-25", end="2020-01-05", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    # Use smaller pore volumes for reasonable residence times (1-3 days)
    aquifer_pore_volumes = np.array([100.0, 200.0, 300.0])

    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Check output type and length
    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(cint_dates)


def test_extraction_to_infiltration_constant_input():
    """Test extraction_to_infiltration with constant output concentration."""
    # Create longer time series for better testing
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Create cin_tedges that properly overlaps with required infiltration times
    # With residence times of 5-10 days, we need cin dates ending around Dec 22-27 to catch Jan 1 cout
    cint_dates = pd.date_range(start="2019-12-15", end="2020-12-25", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.ones(len(dates)) * 5.0, index=dates)  # Constant concentration
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)  # Constant flow
    aquifer_pore_volumes = np.array([500.0, 1000.0])  # 5 and 10 day residence times

    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Explicit validation
    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(cint_dates), f"Expected {len(cint_dates)} output bins, got {len(cin)}"

    valid_count = np.sum(~np.isnan(cin))
    assert valid_count >= 300, f"Expected at least 300 valid bins with proper overlap, got {valid_count}"

    # Test non-negativity and constant preservation
    valid_inputs = cin[~np.isnan(cin)]
    assert np.all(valid_inputs >= 0), "All inputs should be non-negative"

    mean_cin = np.mean(valid_inputs)
    assert abs(mean_cin - 5.0) < 0.5, f"Expected mean ~5.0 (preserved from constant output), got {mean_cin:.3f}"


def test_extraction_to_infiltration_single_pore_volume():
    """Test extraction_to_infiltration with single pore volume."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cint_dates = pd.date_range(start="2019-12-20", end="2020-01-10", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.sin(np.linspace(0, 2 * np.pi, len(dates))) + 2, index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    # Use smaller pore volume for reasonable residence time (5 days)
    aquifer_pore_volumes = np.array([500.0])  # Single pore volume

    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(cint_dates)


def test_extraction_to_infiltration_retardation_factor():
    """Test extraction_to_infiltration with different retardation factors."""
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cint_dates = pd.date_range(start="2019-06-01", end="2019-11-30", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    aquifer_pore_volumes = np.array([1000.0, 2000.0])

    # Test different retardation factors
    cin1 = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    cin2 = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=2.0,
    )

    # Results should be different for different retardation factors
    assert isinstance(cin1, np.ndarray)
    assert isinstance(cin2, np.ndarray)
    assert len(cin1) == len(cin2)


def test_extraction_to_infiltration_error_conditions():
    """Test extraction_to_infiltration error conditions."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)
    aquifer_pore_volumes = np.array([1000.0])

    # Test mismatched tedges length
    wrong_tedges = tedges[:-2]  # Too short
    with pytest.raises(ValueError, match="tedges must have one more element than cout"):
        extraction_to_infiltration(
            cout=cout.values,
            flow=flow.values,
            tedges=wrong_tedges,
            cin_tedges=cin_tedges,
            aquifer_pore_volumes=aquifer_pore_volumes,
        )

    # Test mismatched flow and tedges
    wrong_flow = flow[:-2]  # Too short
    with pytest.raises(ValueError, match="tedges must have one more element than flow"):
        extraction_to_infiltration(
            cout=cout.values,
            flow=wrong_flow.values,
            tedges=tedges,
            cin_tedges=cin_tedges,
            aquifer_pore_volumes=aquifer_pore_volumes,
        )


# ===============================================================================
# PERFECT INVERSE RELATIONSHIP TESTS (MATHEMATICAL SYMMETRY)
# ===============================================================================


def test_extraction_to_infiltration_analytical_simple_delay():
    """Test extraction_to_infiltration with known simple delay scenario."""
    # Create a scenario where we know the exact relationship
    dates = pd.date_range(start="2020-01-01", end="2020-01-20", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Input period starts earlier to account for residence time
    cint_dates = pd.date_range(start="2019-12-25", end="2020-01-15", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    # Known pore volume that gives exactly 1 day residence time
    flow_rate = 100.0  # m3/day
    pore_volume = 100.0  # m3 -> residence time = 100/100 = 1 day

    # Step function: cout jumps from 1 to 5 on day 5
    cout_values = [1.0, 1.0, 1.0, 1.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0]
    cout = pd.Series(cout_values, index=dates)
    flow = pd.Series([flow_rate] * len(dates), index=dates)
    aquifer_pore_volumes = np.array([pore_volume])

    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # With 1-day residence time, the step change on day 5 should appear 1 day earlier in cin
    valid_inputs = cin[~np.isnan(cin)]
    if len(valid_inputs) > 0:
        # Should recover some reasonable signal
        assert np.all(valid_inputs >= 0), f"All inputs should be non-negative, got {valid_inputs}"


def test_extraction_to_infiltration_zero_output_gives_zero_input():
    """Test extraction_to_infiltration with zero output gives zero input."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cint_dates = pd.date_range(start="2019-12-25", end="2020-01-05", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    # Zero concentration everywhere
    cout = pd.Series([0.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)
    aquifer_pore_volumes = np.array([200.0, 400.0])

    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Zero output should give zero input (where valid)
    valid_inputs = cin[~np.isnan(cin)]
    if len(valid_inputs) > 0:
        np.testing.assert_allclose(valid_inputs, 0.0, atol=1e-15, err_msg="Zero output should produce zero input")


# ===============================================================================
# SYMMETRIC EDGE CASE TESTS
# ===============================================================================


def test_extraction_to_infiltration_no_temporal_overlap():
    """Test extraction_to_infiltration returns NaN when no temporal overlap exists."""
    # Create cout in early 2020
    early_dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=early_dates, number_of_bins=len(early_dates))

    # Create cin_tedges much later (no possible overlap)
    late_dates = pd.date_range(start="2020-12-01", end="2020-12-05", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=late_dates, number_of_bins=len(late_dates))

    cout = pd.Series(np.ones(len(early_dates)), index=early_dates)
    flow = pd.Series(np.ones(len(early_dates)) * 100, index=early_dates)
    aquifer_pore_volumes = np.array([100.0])  # Small pore volume

    # No temporal overlap should return all NaN values
    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array of NaN values
    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(late_dates)
    assert np.all(np.isnan(cin))


def test_extraction_to_infiltration_extreme_pore_volumes():
    """Test extraction_to_infiltration handles extremely large pore volumes gracefully."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cint_dates = pd.date_range(start="2019-12-25", end="2020-01-05", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Extremely large pore volumes that create invalid extraction edges
    aquifer_pore_volumes = np.array([1e10, 1e12, 1e15])

    # Should handle extreme pore volumes gracefully
    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array (likely all NaN due to extreme residence times)
    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(cint_dates)
    # With extremely large pore volumes, all outputs should be NaN
    assert np.all(np.isnan(cin))


def test_extraction_to_infiltration_zero_flow():
    """Test extraction_to_infiltration handles zero flow values gracefully."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cint_dates = pd.date_range(start="2019-12-25", end="2020-01-05", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cint_dates, number_of_bins=len(cint_dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.zeros(len(dates)), index=dates)  # Zero flow
    aquifer_pore_volumes = np.array([1000.0])

    # Zero flow creates infinite residence times but should be handled gracefully
    cin = extraction_to_infiltration(
        cout=cout.values,
        flow=flow.values,
        tedges=tedges,
        cin_tedges=cin_tedges,
        aquifer_pore_volumes=aquifer_pore_volumes,
        retardation_factor=1.0,
    )

    # Should return array (likely all NaN due to infinite residence times)
    assert isinstance(cin, np.ndarray)
    assert len(cin) == len(cint_dates)
    # With zero flow, all outputs should be NaN
    assert np.all(np.isnan(cin))


# ===============================================================================
# GAMMA_EXTRACTION_TO_INFILTRATION FUNCTION TESTS
# ===============================================================================


def test_gamma_extraction_to_infiltration_zero_output_gives_zero_input():
    """Test gamma_extraction_to_infiltration with zero output gives zero input."""
    # Use sufficient time span: 60 days extraction, 90 days infiltration window
    dates = pd.date_range(start="2020-01-01", end="2020-03-01", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Infiltration window starts earlier to capture source
    cin_dates = pd.date_range(start="2019-12-01", end="2020-02-28", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    # Zero concentration everywhere
    cout = pd.Series([0.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    cin = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        alpha=5.0,
        beta=40.0,  # mean pore volume = 200 m3, ~2 day residence time with 100 m3/day flow
        n_bins=10,
    )

    # Zero extraction should give zero infiltration (where valid)
    assert len(cin) == len(cin_dates), f"Expected {len(cin_dates)} output bins, got {len(cin)}"
    valid_count = np.sum(~np.isnan(cin))
    assert valid_count >= 50, f"Expected at least 50 valid bins, got {valid_count}"

    valid_inputs = cin[~np.isnan(cin)]
    np.testing.assert_allclose(valid_inputs, 0.0, atol=1e-15, err_msg="Zero extraction must produce zero infiltration")


def test_gamma_extraction_to_infiltration_constant_input():
    """Test constant extraction recovers constant infiltration in fully informed region."""
    # Use long time series: 365 days extraction, extended infiltration window
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    # Infiltration window extends earlier to capture all source contributions
    cin_dates = pd.date_range(start="2019-11-01", end="2020-11-30", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    # Constant extraction concentration
    cout = pd.Series([5.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    cin = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,  # mean pore volume = 100 m3, ~1 day residence time
        n_bins=20,
    )

    # Verify output structure
    assert len(cin) == len(cin_dates), f"Expected {len(cin_dates)} output bins"

    # Count valid bins - should have substantial overlap
    valid_count = np.sum(~np.isnan(cin))
    assert valid_count >= 300, f"Expected at least 300 valid bins for constant signal, got {valid_count}"

    # For constant extraction, infiltration should also be constant in fully informed region
    # Check the middle 200 bins where steady state is guaranteed
    valid_indices = np.where(~np.isnan(cin))[0]
    assert len(valid_indices) >= 300, f"Need at least 300 valid bins, got {len(valid_indices)}"

    # Take middle 200 bins (skip 50 from each end)
    middle_indices = valid_indices[50:-50]
    assert len(middle_indices) >= 200, f"Need at least 200 middle bins, got {len(middle_indices)}"

    middle_values = cin[middle_indices]
    assert not np.any(np.isnan(middle_values)), "Middle region must have no NaN values"

    # Constant extraction must produce constant infiltration
    mean_input = np.mean(middle_values)
    std_input = np.std(middle_values)
    assert abs(mean_input - 5.0) < 0.5, f"Expected mean ~5.0 in steady state, got {mean_input:.3f}"
    assert std_input < 0.5, f"Expected low variance (std < 0.5) in steady state, got {std_input:.3f}"


def test_gamma_extraction_to_infiltration_step_function():
    """Test gamma_extraction_to_infiltration can handle step function in extraction."""
    # Create sufficient time period
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cin_dates = pd.date_range(start="2019-11-01", end="2020-11-30", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    # Step function: extraction concentration changes from 1 to 5
    cout_values = np.ones(len(dates))
    cout_values[180:] = 5.0  # Step on day 180
    cout = pd.Series(cout_values, index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    cin = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,  # ~1 day mean residence time
        n_bins=20,
    )

    # Explicit validation of output quantity
    assert len(cin) == len(cin_dates), f"Expected {len(cin_dates)} output bins, got {len(cin)}"
    valid_count = np.sum(~np.isnan(cin))
    assert valid_count >= 300, f"Expected at least 300 valid bins for year-long data, got {valid_count}"

    # Extract valid values and test variation
    valid_mask = ~np.isnan(cin)
    valid_cin = cin[valid_mask]
    cin_std = np.std(valid_cin)
    assert cin_std > 0.5, f"Expected std > 0.5 to see step variation, got {cin_std:.3f}"

    # Test that both low and high values are present (step function was recovered)
    cin_min = np.min(valid_cin)
    cin_max = np.max(valid_cin)
    assert cin_min < 2.0, f"Expected values below 2.0 (before step), got min={cin_min:.3f}"
    assert cin_max > 3.5, f"Expected values above 3.5 (after step), got max={cin_max:.3f}"


def test_gamma_extraction_to_infiltration_roundtrip():
    """Test gamma_infiltration_to_extraction -> gamma_extraction_to_infiltration roundtrip."""
    # Create time windows with proper alignment - use full year for sufficient overlap
    cin_dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    # Output window overlaps with input
    cout_dates = pd.date_range(start="2020-03-01", end="2020-10-31", freq="D")
    cout_tedges = compute_time_edges(tedges=None, tstart=None, tend=cout_dates, number_of_bins=len(cout_dates))

    # Use varying signal (sine wave) to test actual transport, not just constant recovery
    cin_original_values = 5.0 + 2.0 * np.sin(2 * np.pi * np.arange(len(cin_dates)) / 30.0)
    cin_original = pd.Series(cin_original_values, index=cin_dates)
    flow_cin = pd.Series([100.0] * len(cin_dates), index=cin_dates)
    flow_cout = pd.Series([100.0] * len(cout_dates), index=cout_dates)

    # Forward pass: infiltration -> extraction
    cout = gamma_infiltration_to_extraction(
        cin=cin_original,
        tedges=cin_tedges,
        cout_tedges=cout_tedges,
        flow=flow_cin,
        alpha=10.0,
        beta=10.0,
        n_bins=20,
    )

    # Backward pass: extraction -> infiltration
    cout_series = pd.Series(cout, index=cout_dates)
    cin_reconstructed = gamma_extraction_to_infiltration(
        cout=cout_series,
        tedges=cout_tedges,
        cin_tedges=cin_tedges,
        flow=flow_cout,
        alpha=10.0,
        beta=10.0,
        n_bins=20,
    )

    # Explicit validation of overlap
    valid_mask = ~np.isnan(cin_reconstructed)
    valid_count = np.sum(valid_mask)
    assert valid_count >= 200, f"Expected at least 200 valid bins for substantial overlap, got {valid_count}"

    # Extract middle region explicitly (skip 50 bins on each end)
    valid_indices = np.where(valid_mask)[0]
    assert len(valid_indices) >= 200, f"Need at least 200 valid bins, got {len(valid_indices)}"

    middle_indices = valid_indices[50:-50]
    assert len(middle_indices) >= 100, (
        f"Need at least 100 middle bins for stable region test, got {len(middle_indices)}"
    )

    middle_start = middle_indices[0]
    middle_end = middle_indices[-1] + 1
    middle_region = slice(middle_start, middle_end)

    reconstructed_middle = cin_reconstructed[middle_region]
    original_middle = cin_original.values[middle_region]

    # Test exact recovery in stable middle region with tight tolerance
    np.testing.assert_allclose(
        reconstructed_middle,
        original_middle,
        rtol=0.15,
        err_msg=f"Roundtrip error: expected mean ~{np.mean(original_middle):.2f}, got {np.mean(reconstructed_middle):.2f}",
    )


def test_gamma_extraction_to_infiltration_retardation_factor():
    """Test gamma_extraction_to_infiltration with different retardation factors."""
    dates = pd.date_range(start="2020-01-01", end="2020-12-31", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cin_dates = pd.date_range(start="2019-11-01", end="2020-11-30", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    # Step function
    cout_values = np.ones(len(dates))
    cout_values[180:] = 3.0
    cout = pd.Series(cout_values, index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Test with retardation factor = 1.0
    cin1 = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        retardation_factor=1.0,
        n_bins=20,
    )

    # Test with retardation factor = 2.0 (doubles residence time)
    cin2 = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        alpha=10.0,
        beta=10.0,
        retardation_factor=2.0,
        n_bins=20,
    )

    # Explicit validation of overlap
    valid_mask = ~np.isnan(cin1) & ~np.isnan(cin2)
    valid_count = np.sum(valid_mask)
    assert valid_count >= 250, f"Expected at least 250 valid overlap bins for year-long data, got {valid_count}"

    # Extract valid values for comparison
    cin1_valid = cin1[valid_mask]
    cin2_valid = cin2[valid_mask]

    # The timing of the step should be different - compute max absolute difference
    max_diff = np.max(np.abs(cin1_valid - cin2_valid))
    assert max_diff > 0.5, f"Expected max difference > 0.5 due to shifted timing, got {max_diff:.3f}"

    # Test that spatial variation differs (distribution of values changes with retardation)
    std1 = np.std(cin1_valid)
    std2 = np.std(cin2_valid)
    # At least one should show significant variation from the step function
    max_std = max(std1, std2)
    assert max_std > 0.3, f"Expected max std > 0.3 showing step variation, got {max_std:.3f}"


def test_gamma_extraction_to_infiltration_with_mean_std():
    """Test gamma_extraction_to_infiltration using mean and std parameters."""
    dates = pd.date_range(start="2020-01-01", end="2020-06-30", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cin_dates = pd.date_range(start="2019-12-01", end="2020-05-31", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    cout = pd.Series([3.0] * len(dates), index=dates)
    flow = pd.Series([100.0] * len(dates), index=dates)

    # Use mean/std instead of alpha/beta
    mean = 100.0  # mean pore volume
    std = 20.0

    cin = gamma_extraction_to_infiltration(
        cout=cout,
        tedges=tedges,
        cin_tedges=cin_tedges,
        flow=flow,
        mean=mean,
        std=std,
        n_bins=20,
    )

    # Explicit validation
    assert len(cin) == len(cin_dates), f"Expected {len(cin_dates)} output bins, got {len(cin)}"
    valid_count = np.sum(~np.isnan(cin))
    assert valid_count >= 100, f"Expected at least 100 valid bins for 6-month data, got {valid_count}"

    # Test mean preservation for constant input
    valid_mask = ~np.isnan(cin)
    valid_cin = cin[valid_mask]
    mean_cin = np.mean(valid_cin)
    assert abs(mean_cin - 3.0) < 0.5, f"Expected mean ~3.0 (preserved from constant input), got {mean_cin:.3f}"

    # For constant input, expect low variation in output
    std_cin = np.std(valid_cin)
    assert std_cin < 0.5, f"Expected low std (<0.5) for constant input, got {std_cin:.3f}"


def test_gamma_extraction_to_infiltration_missing_parameters():
    """Test that gamma_extraction_to_infiltration raises appropriate errors for missing parameters."""
    dates = pd.date_range(start="2020-01-01", end="2020-01-10", freq="D")
    tedges = compute_time_edges(tedges=None, tstart=None, tend=dates, number_of_bins=len(dates))

    cin_dates = pd.date_range(start="2019-12-28", end="2020-01-08", freq="D")
    cin_tedges = compute_time_edges(tedges=None, tstart=None, tend=cin_dates, number_of_bins=len(cin_dates))

    cout = pd.Series(np.ones(len(dates)), index=dates)
    flow = pd.Series(np.ones(len(dates)) * 100, index=dates)

    # Test missing both alpha/beta and mean/std
    with pytest.raises(ValueError):
        gamma_extraction_to_infiltration(cout=cout, tedges=tedges, cin_tedges=cin_tedges, flow=flow)
