"""DSPy signatures and modules used by the evaluator."""

import dspy


class QuerySignature(dspy.Signature):
    """A signature that takes a complete user request."""

    request = dspy.InputField(
        desc="The user's full request, including all instructions, context, and relevant code snippets."
    )
    guidelines = dspy.InputField(
        desc="Optional long-form documentation or rules to follow as background context."
    )

    answer = dspy.OutputField(
        desc="The final, complete answer that directly fulfills the request."
    )

class QualityGrader(dspy.Signature):
    """
    You are an expert evaluator. Your goal is to grade the generated_answer based on how well it achieves the expected_outcome for the original task_requirements.

    Use the reference_answer as a gold standard for a high-quality response. The generated_answer does not need to match it verbatim, but it should capture the key points and follow the same spirit.

    Be concise and focused in your evaluation. Provide succinct, specific feedback rather than verbose explanations.
    """
    expected_outcome = dspy.InputField(desc="The high-level success criterion that the generated_answer should achieve.")
    request = dspy.InputField(desc="The original user request to be analyzed.")
    reference_answer = dspy.InputField(desc="The ground truth or ideal answer used as a gold standard for comparison.")
    generated_answer = dspy.InputField(desc="The answer generated by the model being evaluated.")
    
    score = dspy.OutputField(desc="A single float score between 0.0 and 1.0.")
    hits = dspy.OutputField(desc="Brief, specific list of what the generated_answer did well or correctly achieved from the expected_outcome.")
    misses = dspy.OutputField(desc="Brief, specific list of what the generated_answer missed, did incorrectly, or failed to achieve from the expected_outcome. Leave completely empty if there are no significant misses.")
    reasoning = dspy.OutputField(desc="Concise explanation for the score (1-2 sentences max).")

class EvaluationModule(dspy.Module):
    """A generic evaluation module that can use any provided signature."""

    def __init__(self, signature_class: dspy.Signature):
        super().__init__()
        self.predictor = dspy.Predict(signature_class)

    def forward(self, test_case_id: str = None, **kwargs):
        """Forwards the request to the configured model's execution method."""
        return dspy.settings.lm.execute_prediction(self, test_case_id=test_case_id, **kwargs)
